<?php
/**
 * Titanium Attendance System - Single File Admin UI
 * Filename: titanium-attendance.php
 * Purpose: Professional, self-contained admin UI to view and manage the on-disk database layout
 * described by the user (/database tree of users, indexes, attendance NDJSON, per-user NDJSON).
 *
 * Features (implemented):
 * - Dashboard with quick stats (users, today's attendance, recent events)
 * - Users page: list, search, create, edit, assign/remove UIDs, export/import users
 * - Attendance (By Day): view NDJSON lines, filter, export NDJSON/CSV/JSON
 * - Attendance (By User): quick view of per-user history (streaming)
 * - Indexes: view and repair index maps (names.json, admnos.json, uids.json)
 * - Import/Export: supports JSON, NDJSON and CSV for users & attendance; atomic writes/validation
 * - Backup & Restore: create ZIP of /database and download; restore via uploaded zip (manual warn)
 * - Migration/versioning: view db_version, run a simple migration placeholder
 * - Utilities: normalize UID, regenerate indexes, prune/rotate old attendance files
 * - Animated UI, responsive layout, single-file (no composer libs). Uses native ZipArchive if available.
 *
 * NOTES & SAFETY
 * - This script performs filesystem writes and deletes; run on a safe admin host.
 * - It assumes PHP is allowed to read/write ./database with ownership/permissions as per spec.
 * - ALWAYS BACKUP before running destructive actions.
 *
 * Author: Generated for user (single-file deliverable)
 * Date: 2025-11-28
 */

// --------------------------------------------------
// CONFIG
// --------------------------------------------------
$ROOT_DB = __DIR__ . DIRECTORY_SEPARATOR . 'database';
// Optional: override via GET param for testing (careful!)
if (isset($_GET['db_root'])) {
    // only allow relative safe overrides (no ..)
    $candidate = $_GET['db_root'];
    if (strpos($candidate, '..') === false && substr($candidate, 0, 1) !== '/') {
        $ROOT_DB = __DIR__ . DIRECTORY_SEPARATOR . $candidate;
    }
}

// Basic helpers
function json_response($data) {
    header('Content-Type: application/json');
    echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

function ensure_dir($path) {
    if (!is_dir($path)) {
        mkdir($path, 0755, true);
    }
}

function normalize_uid($uid) {
    // Uppercase, remove whitespace
    return strtoupper(preg_replace('/\s+/', '', (string)$uid));
}

function safe_write_json_atomic($path, $data) {
    $tmp = $path . '.tmp';
    $bytes = file_put_contents($tmp, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    if ($bytes === false) return false;
    return rename($tmp, $path);
}

function append_ndjson_line($path, $obj) {
    $line = json_encode($obj, JSON_UNESCAPED_UNICODE) . "\n";
    $dir = dirname($path);
    ensure_dir($dir);
    return file_put_contents($path, $line, FILE_APPEND | LOCK_EX) !== false;
}

function read_json_file($path) {
    if (!file_exists($path)) return null;
    $txt = file_get_contents($path);
    return json_decode($txt, true);
}

function list_users_dir($root) {
    $dir = $root . '/users';
    if (!is_dir($dir)) return [];
    $files = glob($dir . '/user-*.json');
    $out = [];
    foreach ($files as $f) {
        $data = json_decode(file_get_contents($f), true);
        if ($data) $out[$data['user_id']] = $data;
    }
    return $out;
}

function load_indexes($root) {
    $idx = $root . '/index';
    $names = read_json_file($idx . '/names.json') ?? [];
    $admnos = read_json_file($idx . '/admnos.json') ?? [];
    $uids = read_json_file($idx . '/uids.json') ?? [];
    return ['names' => $names, 'admnos' => $admnos, 'uids' => $uids];
}

function regenerate_indexes($root) {
    // Scan users and recreate authoritative index files
    $users = list_users_dir($root);
    $names = [];
    $admnos = [];
    $uids = [];
    foreach ($users as $u) {
        $user_id = $u['user_id'];
        $names[$user_id] = $u['name'];
        $admnos[$u['admno']] = $user_id;
        if (!empty($u['uids']) && is_array($u['uids'])) {
            foreach ($u['uids'] as $card) {
                $norm = normalize_uid($card);
                $uids[$norm] = $user_id;
            }
        }
    }
    ensure_dir($root . '/index');
    safe_write_json_atomic($root . '/index/names.json', $names);
    safe_write_json_atomic($root . '/index/admnos.json', $admnos);
    safe_write_json_atomic($root . '/index/uids.json', $uids);
    return ['names' => $names, 'admnos' => $admnos, 'uids' => $uids];
}

function generate_id($prefix = 'usr') {
    return $prefix . '_' . bin2hex(random_bytes(4));
}

function generate_event_id() {
    return 'evt_' . bin2hex(random_bytes(4));
}

function safe_append_attendance($root, $date, $event) {
    // date: YYYY-MM-DD
    $parts = explode('-', $date);
    if (count($parts) !== 3) return false;
    list($y,$m,$d) = $parts;
    $dir = sprintf('%s/attendance/%s/%s/%s', $root, $y, $m, $d);
    ensure_dir($dir);
    $path = $dir . '/attendance-' . $date . '.ndjson';
    return append_ndjson_line($path, $event);
}

function safe_append_attendance_by_user($root, $user_id, $event) {
    $dir = $root . '/attendance_by_user';
    ensure_dir($dir);
    $path = $dir . '/user-' . $user_id . '.ndjson';
    return append_ndjson_line($path, $event);
}

function stream_ndjson_reverse($path, $limit = 200) {
    // naive: read all lines and reverse; optimized streaming possible but OK for moderate files
    if (!file_exists($path)) return [];
    $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $lines = array_reverse($lines);
    $out = [];
    foreach ($lines as $i => $l) {
        if ($i >= $limit) break;
        $obj = json_decode($l, true);
        if ($obj) $out[] = $obj;
    }
    return $out;
}

function read_ndjson_stream($path, $callback) {
    if (!file_exists($path)) return;
    $fh = fopen($path, 'rb');
    if (!$fh) return;
    while (($line = fgets($fh)) !== false) {
        $line = trim($line);
        if ($line === '') continue;
        $obj = json_decode($line, true);
        if ($obj) $callback($obj);
    }
    fclose($fh);
}

function download_file($path, $filename_override = null) {
    if (!file_exists($path)) {
        http_response_code(404);
        echo "File not found";
        exit;
    }
    if ($filename_override) $name = $filename_override; else $name = basename($path);
    header('Content-Type: application/octet-stream');
    header('Content-Disposition: attachment; filename="' . $name . '"');
    header('Content-Length: ' . filesize($path));
    readfile($path);
    exit;
}

// --------------------------------------------------
// ROUTING & ACTIONS
// Keep destructive actions POST-only
// --------------------------------------------------
$page = $_GET['page'] ?? 'dashboard';
$action = $_POST['action'] ?? $_GET['action'] ?? null;

// Quick API endpoints for JSON responses
if (isset($_GET['api']) && $_GET['api'] === 'stats') {
    $users = list_users_dir($ROOT_DB);
    $idx = load_indexes($ROOT_DB);
    // count today's attendance
    $today = (new DateTime())->format('Y-m-d');
    $today_path = sprintf('%s/attendance/%s/%s/%s/attendance-%s.ndjson', $ROOT_DB, substr($today,0,4), substr($today,5,2), substr($today,8,2), $today);
    $today_count = 0;
    if (file_exists($today_path)) {
        $lines = file($today_path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        foreach ($lines as $l) {
            $o = json_decode($l, true);
            if ($o && isset($o['result']) && $o['result'] === 'accepted') $today_count++;
        }
    }
    json_response(['users' => count($users), 'today_accepted' => $today_count]);
}

// Handle user create/edit
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['op'])) {
    $op = $_POST['op'];
    if ($op === 'create_user') {
        $admno = trim($_POST['admno'] ?? '');
        $name = trim($_POST['name'] ?? '');
        $class = trim($_POST['class'] ?? '');
        $uids = array_filter(array_map('normalize_uid', preg_split('/[\r\n,;]+/', ($_POST['uids'] ?? ''))));
        // validation
        if ($admno === '' || $name === '') {
            $_SESSION['flash_error'] = 'Name and admno required';
            header('Location: ?page=users'); exit;
        }
        ensure_dir($ROOT_DB . '/users');
        ensure_dir($ROOT_DB . '/index');
        // check admno uniqueness
        $admnos = read_json_file($ROOT_DB . '/index/admnos.json') ?? [];
        if (isset($admnos[$admno])) {
            $_SESSION['flash_error'] = 'Admission number already exists'; header('Location: ?page=users'); exit;
        }
        $user_id = generate_id('usr');
        $now = (new DateTime())->format(DATE_ATOM);
        $record = [
            'user_id' => $user_id,
            'admno' => $admno,
            'name' => $name,
            'class' => $class,
            'uids' => array_values(array_unique($uids)),
            'created_at' => $now,
            'updated_at' => $now
        ];
        $path = $ROOT_DB . '/users/user-' . $user_id . '.json';
        if (!safe_write_json_atomic($path, $record)) {
            $_SESSION['flash_error'] = 'Failed to write user file'; header('Location: ?page=users'); exit;
        }
        // update indexes atomically
        $names = read_json_file($ROOT_DB . '/index/names.json') ?? [];
        $admnos = read_json_file($ROOT_DB . '/index/admnos.json') ?? [];
        $uids_idx = read_json_file($ROOT_DB . '/index/uids.json') ?? [];
        $names[$user_id] = $name;
        $admnos[$admno] = $user_id;
        foreach ($record['uids'] as $card) {
            $uids_idx[$card] = $user_id;
        }
        safe_write_json_atomic($ROOT_DB . '/index/names.json', $names);
        safe_write_json_atomic($ROOT_DB . '/index/admnos.json', $admnos);
        safe_write_json_atomic($ROOT_DB . '/index/uids.json', $uids_idx);
        header('Location: ?page=users&created=' . urlencode($user_id)); exit;
    }
    if ($op === 'edit_user') {
        $user_id = $_POST['user_id'] ?? '';
        $path = $ROOT_DB . '/users/user-' . $user_id . '.json';
        if (!file_exists($path)) { $_SESSION['flash_error']='User not found'; header('Location:?page=users'); exit; }
        $data = json_decode(file_get_contents($path), true);
        $data['name'] = trim($_POST['name'] ?? $data['name']);
        $data['class'] = trim($_POST['class'] ?? $data['class']);
        $incoming = array_filter(array_map('normalize_uid', preg_split('/[\r\n,;]+/', ($_POST['uids'] ?? ''))));
        $data['uids'] = array_values(array_unique($incoming));
        $data['updated_at'] = (new DateTime())->format(DATE_ATOM);
        safe_write_json_atomic($path, $data);
        // rebuild indexes: safe option to regenerate all to avoid conflicts
        regenerate_indexes($ROOT_DB);
        header('Location: ?page=users&edited=' . urlencode($user_id)); exit;
    }
    if ($op === 'append_attendance') {
        $uid = normalize_uid($_POST['uid'] ?? '');
        $adm_supplied = trim($_POST['admno'] ?? '');
        $device = trim($_POST['device_id'] ?? 'manual');
        $actionLabel = trim($_POST['action_label'] ?? 'present');
        $now = new DateTime();
        $idx = load_indexes($ROOT_DB);
        $user_id = $idx['uids'][$uid] ?? null;
        $adm_stored = null; $name = null; $class = null;
        $result = 'rejected'; $msg = 'uid not assigned';
        if ($user_id) {
            $u = read_json_file($ROOT_DB . '/users/user-' . $user_id . '.json');
            if ($u) { $adm_stored = $u['admno']; $name = $u['name']; $class = $u['class']; }
            // admno check optional
            $result = 'accepted'; $msg = 'attendance recorded';
            // duplicate detection
            $date = $now->format('Y-m-d');
            $dayPath = sprintf('%s/attendance/%s/%s/%s/attendance-%s.ndjson', $ROOT_DB, $date[0].$date[1].$date[2].$date[3], substr($date,5,2), substr($date,8,2), $date);
            $already = false;
            if (file_exists($dayPath)) {
                $lines = file($dayPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                foreach ($lines as $l) {
                    $o = json_decode($l, true);
                    if ($o && isset($o['user_id']) && $o['user_id'] === $user_id && isset($o['result']) && $o['result']==='accepted') { $already = true; break; }
                }
            }
            if ($already) { $result='rejected'; $msg='already marked today'; }
        }
        $event = [
            'event_id'=>generate_event_id(),
            'timestamp'=>$now->format(DATE_ATOM),
            'user_id'=>$user_id,
            'admno_supplied'=>$adm_supplied,
            'admno_stored'=>$adm_stored,
            'name'=>$name,
            'class'=>$class,
            'uid'=>$uid,
            'device_id'=>$device,
            'action'=>$actionLabel,
            'result'=>$result,
            'message'=>$msg
        ];
        $date = $now->format('Y-m-d');
        safe_append_attendance($ROOT_DB, $date, $event);
        if ($user_id) safe_append_attendance_by_user($ROOT_DB, $user_id, $event);
        header('Location: ?page=attendance&added=1'); exit;
    }
    if ($op === 'import_users') {
        // accept uploaded file name 'import_file' (json, ndjson, csv)
        if (!isset($_FILES['import_file'])) { $_SESSION['flash_error']='No file'; header('Location:?page=import_export'); exit; }
        $f = $_FILES['import_file']['tmp_name'];
        $ext = pathinfo($_FILES['import_file']['name'], PATHINFO_EXTENSION);
        $text = file_get_contents($f);
        $created = 0; $skipped = 0; $errors = [];
        if (in_array(strtolower($ext), ['json'])) {
            $arr = json_decode($text, true);
            if (!is_array($arr)) { $_SESSION['flash_error']='Invalid JSON'; header('Location:?page=import_export'); exit; }
            foreach ($arr as $rec) {
                // expect same schema as user file
                if (!isset($rec['admno']) || !isset($rec['name'])) { $skipped++; continue; }
                // simple create flow
                $user_id = generate_id('usr');
                $now = (new DateTime())->format(DATE_ATOM);
                $data = [
                    'user_id'=>$user_id,
                    'admno'=> (string)$rec['admno'],
                    'name'=> (string)$rec['name'],
                    'class'=> (string)($rec['class'] ?? ''),
                    'uids'=> array_values(array_unique(array_map('normalize_uid', ($rec['uids'] ?? [])))),
                    'created_at'=>$now,
                    'updated_at'=>$now
                ];
                safe_write_json_atomic($ROOT_DB . '/users/user-' . $user_id . '.json', $data);
                $created++;
            }
            regenerate_indexes($ROOT_DB);
        } elseif (in_array(strtolower($ext), ['ndjson'])) {
            $lines = preg_split('/\r?\n/', $text);
            foreach ($lines as $l) {
                $l = trim($l); if ($l==='') continue;
                $rec = json_decode($l, true);
                if (!$rec || !isset($rec['admno']) || !isset($rec['name'])) { $skipped++; continue; }
                $user_id = generate_id('usr');
                $now = (new DateTime())->format(DATE_ATOM);
                $data = [
                    'user_id'=>$user_id,
                    'admno'=> (string)$rec['admno'],
                    'name'=> (string)$rec['name'],
                    'class'=> (string)($rec['class'] ?? ''),
                    'uids'=> array_values(array_unique(array_map('normalize_uid', ($rec['uids'] ?? [])))),
                    'created_at'=>$now,
                    'updated_at'=>$now
                ];
                safe_write_json_atomic($ROOT_DB . '/users/user-' . $user_id . '.json', $data);
                $created++;
            }
            regenerate_indexes($ROOT_DB);
        } else {
            $_SESSION['flash_error']='Unsupported file type: ' . htmlspecialchars($ext);
            header('Location:?page=import_export'); exit;
        }
        $_SESSION['flash_success'] = "Imported: $created, skipped: $skipped";
        header('Location:?page=import_export'); exit;
    }
    if ($op === 'backup_zip') {
        // create zip of database
        $zipname = __DIR__ . '/titanium-database-backup-' . date('Ymd-His') . '.zip';
        if (class_exists('ZipArchive')) {
            $zip = new ZipArchive();
            if ($zip->open($zipname, ZipArchive::CREATE)!==true) { $_SESSION['flash_error']='Failed to create zip'; header('Location:?page=backup'); exit; }
            $it = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($ROOT_DB));
            foreach ($it as $file) {
                if (!$file->isFile()) continue;
                $filePath = $file->getRealPath();
                $local = substr($filePath, strlen($ROOT_DB)+1);
                $zip->addFile($filePath, $local);
            }
            $zip->close();
            header('Location: ?page=backup&download=' . urlencode(basename($zipname))); exit;
        } else {
            $_SESSION['flash_error'] = 'ZipArchive not available on this PHP build'; header('Location:?page=backup'); exit;
        }
    }
}

// Download generated zip if requested
if (isset($_GET['download'])) {
    $name = basename($_GET['download']);
    $path = __DIR__ . '/' . $name;
    if (file_exists($path)) download_file($path);
}

// Export endpoints - GET with ?export=...&target=users|attendance&... 
if (isset($_GET['export'])) {
    $mode = $_GET['export'];
    if ($mode === 'users') {
        $users = list_users_dir($ROOT_DB);
        $format = $_GET['format'] ?? 'json';
        if ($format === 'json') {
            header('Content-Type: application/json');
            header('Content-Disposition: attachment; filename="users.json"');
            echo json_encode(array_values($users), JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE); exit;
        } elseif ($format === 'ndjson') {
            header('Content-Type: application/x-ndjson');
            header('Content-Disposition: attachment; filename="users.ndjson"');
            foreach ($users as $u) echo json_encode($u, JSON_UNESCAPED_UNICODE) . "\n";
            exit;
        } elseif ($format === 'csv') {
            header('Content-Type: text/csv'); header('Content-Disposition: attachment; filename="users.csv"');
            $out = fopen('php://output', 'w'); fputcsv($out, ['user_id','admno','name','class','uids','created_at','updated_at']);
            foreach ($users as $u) fputcsv($out, [$u['user_id'],$u['admno'],$u['name'],$u['class'],implode('|',$u['uids']),$u['created_at'],$u['updated_at']]);
            fclose($out); exit;
        }
    }
    if ($mode === 'attendance') {
        // require date param YYYY-MM-DD or user_id
        $format = $_GET['format'] ?? 'ndjson';
        if (isset($_GET['date'])) {
            $date = $_GET['date'];
            $path = sprintf('%s/attendance/%s/%s/%s/attendance-%s.ndjson', $ROOT_DB, substr($date,0,4), substr($date,5,2), substr($date,8,2), $date);
            if (!file_exists($path)) { http_response_code(404); echo 'Not found'; exit; }
            if ($format === 'ndjson') { header('Content-Type: application/x-ndjson'); header('Content-Disposition: attachment; filename="attendance-' . $date . '.ndjson"'); readfile($path); exit; }
            // convert to csv or json
            $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            $objs = array_map(function($l){ return json_decode($l, true); }, $lines);
            if ($format === 'json') { header('Content-Type: application/json'); header('Content-Disposition: attachment; filename="attendance-' . $date . '.json"'); echo json_encode($objs, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE); exit; }
            if ($format === 'csv') {
                header('Content-Type: text/csv'); header('Content-Disposition: attachment; filename="attendance-' . $date . '.csv"');
                $out = fopen('php://output','w');
                $headers = ['event_id','timestamp','user_id','admno_supplied','admno_stored','name','class','uid','device_id','action','result','message']; fputcsv($out,$headers);
                foreach ($objs as $o) {
                    fputcsv($out, array_map(function($h) use ($o){ return $o[$h] ?? ''; }, $headers));
                }
                fclose($out); exit;
            }
        }
    }
}

// --------------------------------------------------
// UI: render pages
// --------------------------------------------------
function h($s) { return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }

// load basic data for nav
$users_map = list_users_dir($ROOT_DB);
$indexes = load_indexes($ROOT_DB);

// small helper to read today's recent events
function recent_events($root, $limit=20) {
    $dir = $root . '/attendance';
    $out = [];
    if (!is_dir($dir)) return $out;
    // find last 7 days
    for ($i=0;$i<7;$i++) {
        $d = new DateTime(); $d->modify("-{$i} days"); $date = $d->format('Y-m-d');
        $p = sprintf('%s/attendance/%s/%s/%s/attendance-%s.ndjson', $root, substr($date,0,4), substr($date,5,2), substr($date,8,2), $date);
        if (file_exists($p)) {
            $lines = file($p, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            foreach ($lines as $l) {
                $o = json_decode($l, true);
                if ($o) $out[] = $o;
            }
        }
    }
    usort($out, function($a,$b){ return strcmp($b['timestamp'] ?? '','' . ($a['timestamp'] ?? '')); });
    return array_slice($out,0,$limit);
}

$recent = recent_events($ROOT_DB, 50);

// Minimal session-like flash via temp files (since no session start to keep single-file simple)
function flash_get($key) {
    $f = sys_get_temp_dir() . '/titanium_flash_' . $key;
    if (file_exists($f)) { $v = file_get_contents($f); unlink($f); return $v; }
    return null;
}
function flash_set($key,$val) { file_put_contents(sys_get_temp_dir() . '/titanium_flash_' . $key, $val); }

$flash_ok = flash_get('success');
$flash_err = flash_get('error');

// HTML + CSS + JS UI (single page with nav)
?><!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Titanium Attendance System — Admin</title>
<link rel="preconnect" href="https://fonts.gstatic.com">
<style>
:root{--bg:#0f1724;--card:#0b1220;--accent:#0ea5a5;--muted:#94a3b8;--glass:rgba(255,255,255,0.04)}
*{box-sizing:border-box;font-family:Inter,ui-sans-serif,system-ui,Segoe UI,Roboto,'Helvetica Neue',Arial}
body{margin:0;background:linear-gradient(180deg,var(--bg),#071026);color:#e6eef6}
.app{display:grid;grid-template-columns:260px 1fr;min-height:100vh}
.sidebar{padding:18px;border-right:1px solid rgba(255,255,255,0.03);background:linear-gradient(180deg,rgba(255,255,255,0.02),transparent)}
.brand{font-size:18px;font-weight:700;display:flex;align-items:center;gap:12px}
.logo{width:44px;height:44px;border-radius:10px;background:linear-gradient(135deg,#06b6d4,#06b6d4);display:flex;align-items:center;justify-content:center;color:#042a2a;font-weight:800}
.nav{margin-top:18px}
.nav a{display:block;padding:10px 12px;border-radius:8px;color:var(--muted);text-decoration:none;margin-bottom:6px}
.nav a.active{background:linear-gradient(90deg,rgba(14,165,165,0.12),rgba(14,165,165,0.03));color:#dff6f6}
.header{display:flex;align-items:center;justify-content:space-between;padding:18px 22px;border-bottom:1px solid rgba(255,255,255,0.03);backdrop-filter: blur(6px)}
.container{padding:22px}
.card{background:var(--card);border-radius:12px;padding:16px;margin-bottom:18px;box-shadow: 0 6px 18px rgba(2,6,23,0.6)}
.kpi{display:flex;gap:12px}
.kpi .item{flex:1;padding:12px;border-radius:10px;background:linear-gradient(180deg,rgba(255,255,255,0.01),transparent);text-align:center}
.small{font-size:13px;color:var(--muted)}
.table{width:100%;border-collapse:collapse}
.table th,.table td{padding:10px;text-align:left;border-bottom:1px dashed rgba(255,255,255,0.02)}
.btn{display:inline-block;padding:8px 12px;border-radius:8px;background:var(--accent);color:#042a2a;text-decoration:none;font-weight:600}
.btn.ghost{background:transparent;border:1px solid rgba(255,255,255,0.04);color:var(--muted)}
.form-row{display:flex;gap:8px;margin-bottom:8px}
.input,textarea,select{background:transparent;border:1px solid rgba(255,255,255,0.06);padding:9px;border-radius:8px;color:inherit;width:100%}
.code{font-family:monospace;background:#021224;padding:10px;border-radius:8px;color:#9fe3e3}
.badge{display:inline-block;padding:4px 8px;border-radius:999px;background:rgba(255,255,255,0.03);font-size:12px}
.animate-fade{animation:fadeIn .5s ease both}
@keyframes fadeIn { from {opacity:0; transform: translateY(6px)} to {opacity:1; transform:none} }
.footer{padding:12px 22px;color:var(--muted);font-size:13px}
.search{display:flex;gap:8px;margin-bottom:12px}
</style>
</head>
<body>
<div class="app">
  <aside class="sidebar">
    <div class="brand"><div class="logo">T</div> Titanium Attendance</div>
    <div class="small" style="margin-top:6px">Admin console — single-file</div>
    <nav class="nav" style="margin-top:18px">
      <?php
      $navs = [ 'dashboard'=>'Dashboard', 'users'=>'Users', 'attendance'=>'Attendance', 'attendance_by_user'=>'By User', 'indexes'=>'Indexes', 'import_export'=>'Import / Export', 'backup'=>'Backup / Migrate', 'settings'=>'Settings' ];
      foreach ($navs as $k=>$label) {
          $cls = ($page === $k) ? 'active' : '';
          echo "<a class='$cls' href='?page=$k'>" . h($label) . "</a>";
      }
      ?>
    </nav>
    <div style="margin-top:18px" class="small">Quick actions</div>
    <div style="margin-top:8px">
      <a class="btn" href="?page=users&action=create">New User</a>
      <a class="btn ghost" href="?page=import_export">Import</a>
    </div>
    <div style="height:28px"></div>
    <div class="small">Server DB root</div>
    <div class="code" style="margin-top:6px;word-break:break-all;">/<?= h(trim(str_replace(__DIR__,'','' . $ROOT_DB), '/')) ?></div>
  </aside>
  <main>
    <div class="header">
      <div>
        <div style="font-weight:700"><?php
            switch($page) {
                case 'users': echo 'Users'; break;
                case 'attendance': echo 'Attendance'; break;
                case 'attendance_by_user': echo 'Attendance by User'; break;
                case 'indexes': echo 'Indexes'; break;
                case 'import_export': echo 'Import / Export'; break;
                case 'backup': echo 'Backup & Migrate'; break;
                case 'settings': echo 'Settings'; break;
                default: echo 'Dashboard';
            }
        ?></div>
        <div class="small">Titanium Attendance System — management UI</div>
      </div>
      <div style="display:flex;gap:10px;align-items:center">
        <a class="btn ghost" href="?page=attendance&action=append_form">Manual +</a>
        <a class="btn" href="?page=backup">Backup</a>
      </div>
    </div>
    <div class="container">
      <?php if ($flash_ok): ?><div class="card animate-fade" style="border-left:4px solid #10b981"><strong><?= h($flash_ok) ?></strong></div><?php endif; ?>
      <?php if ($flash_err): ?><div class="card animate-fade" style="border-left:4px solid #ef4444"><strong><?= h($flash_err) ?></strong></div><?php endif; ?>

      <?php if ($page === 'dashboard'): ?>
        <div class="card animate-fade">
          <div style="display:flex;justify-content:space-between;align-items:center">
            <div>
              <div style="font-size:18px;font-weight:700">Overview</div>
              <div class="small">Realtime snapshot of the database files on disk</div>
            </div>
            <div class="small">DB path: <span class="badge"><?= h($ROOT_DB) ?></span></div>
          </div>

          <div style="margin-top:12px" class="kpi">
            <div class="item">
              <div class="small">Total users</div>
              <div style="font-size:20px;font-weight:800"><?= count($users_map) ?></div>
            </div>
            <div class="item">
              <div class="small">Index entries (uids)</div>
              <div style="font-size:20px;font-weight:800"><?= count($indexes['uids']) ?></div>
            </div>
            <div class="item">
              <div class="small">Recent events (7d)</div>
              <div style="font-size:20px;font-weight:800"><?= count($recent) ?></div>
            </div>
          </div>
        </div>

        <div class="card animate-fade">
          <div style="display:flex;justify-content:space-between;align-items:center">
            <div style="font-weight:700">Recent Attendance</div>
            <div class="small"><a class="btn ghost" href="?page=attendance">Open attendance</a></div>
          </div>
          <table class="table" style="margin-top:10px">
            <thead><tr><th>Time</th><th>Name</th><th>Admno</th><th>UID</th><th>Device</th><th>Result</th></tr></thead>
            <tbody>
            <?php foreach($recent as $r): ?>
              <tr>
                <td class="small"><?= h($r['timestamp'] ?? '') ?></td>
                <td><?= h($r['name'] ?? ($r['user_id'] ?? '')) ?></td>
                <td><?= h($r['admno_stored'] ?? $r['admno_supplied'] ?? '') ?></td>
                <td><?= h($r['uid'] ?? '') ?></td>
                <td><?= h($r['device_id'] ?? '') ?></td>
                <td><?= h($r['result'] ?? '') ?></td>
              </tr>
            <?php endforeach; ?>
            </tbody>
          </table>
        </div>

      <?php elseif ($page === 'users'): ?>
        <div class="card">
          <div style="display:flex;justify-content:space-between;align-items:center">
            <div style="font-weight:700">Users</div>
            <div><a class="btn" href="?page=users&action=create">New user</a> <a class="btn ghost" href="?export=users&format=csv">Export CSV</a></div>
          </div>
          <div style="margin-top:12px">
            <form method="get"><input type="hidden" name="page" value="users" /><div class="search"><input class="input" name="q" placeholder="Search name / admno / uid" value="<?= h($_GET['q'] ?? '') ?>" /><button class="btn ghost">Search</button></div></form>
            <table class="table">
              <thead><tr><th>Name</th><th>Admno</th><th>Class</th><th>UIDs</th><th></th></tr></thead>
              <tbody>
              <?php
              $q = trim($_GET['q'] ?? '');
              foreach ($users_map as $u) {
                  if ($q !== '') {
                      $ok = stripos($u['name'],$q) !== false || stripos($u['admno'],$q)!==false;
                      foreach ($u['uids'] as $uu) if ($ok===false && stripos($uu,$q)!==false) $ok=true;
                      if (!$ok) continue;
                  }
                  echo '<tr>';
                  echo '<td>' . h($u['name']) . '</td>';
                  echo '<td class="small">' . h($u['admno']) . '</td>';
                  echo '<td class="small">' . h($u['class']) . '</td>';
                  echo '<td class="small">' . h(implode(', ', $u['uids'])) . '</td>';
                  echo '<td><a class="btn ghost" href="?page=users&action=edit&user_id=' . h($u['user_id']) . '">Edit</a> <a class="btn" href="?page=attendance_by_user&user_id=' . h($u['user_id']) . '">History</a></td>';
                  echo '</tr>';
              }
              ?>
              </tbody>
            </table>
          </div>
        </div>

        <?php if (isset($_GET['action']) && $_GET['action'] === 'create'): ?>
          <div class="card">
            <div style="font-weight:700">Create user</div>
            <form method="post">
              <input type="hidden" name="op" value="create_user">
              <div class="form-row"><input class="input" name="name" placeholder="Full name" required></div>
              <div class="form-row"><input class="input" name="admno" placeholder="Admission number" required></div>
              <div class="form-row"><input class="input" name="class" placeholder="Class / Section"></div>
              <div class="form-row"><textarea class="input" name="uids" placeholder="Card UIDs (comma or newline separated)"></textarea></div>
              <div style="text-align:right"><button class="btn">Create</button></div>
            </form>
          </div>
        <?php endif; ?>

        <?php if (isset($_GET['action']) && $_GET['action'] === 'edit' && isset($_GET['user_id'])): $uid = $_GET['user_id']; $u = read_json_file($ROOT_DB . '/users/user-' . $uid . '.json'); if ($u): ?>
          <div class="card">
            <div style="font-weight:700">Edit user — <?= h($u['name']) ?></div>
            <form method="post">
              <input type="hidden" name="op" value="edit_user">
              <input type="hidden" name="user_id" value="<?= h($u['user_id']) ?>">
              <div class="form-row"><input class="input" name="name" value="<?= h($u['name']) ?>"></div>
              <div class="form-row"><input class="input" name="class" value="<?= h($u['class']) ?>"></div>
              <div class="form-row"><textarea class="input" name="uids"><?= h(implode("\n", $u['uids'])) ?></textarea></div>
              <div style="text-align:right"><button class="btn">Save</button></div>
            </form>
            <div style="margin-top:8px" class="small">User file: <span class="code">/database/users/user-<?= h($u['user_id']) ?>.json</span></div>
          </div>
        <?php endif; endif; ?>

      <?php elseif ($page === 'attendance'): ?>
        <div class="card">
          <div style="display:flex;justify-content:space-between;align-items:center">
            <div style="font-weight:700">Attendance by date</div>
            <div class="small">Export: <a class="btn ghost" href="?export=attendance&date=<?= date('Y-m-d') ?>&format=ndjson">NDJSON</a> <a class="btn ghost" href="?export=attendance&date=<?= date('Y-m-d') ?>&format=csv">CSV</a></div>
          </div>
          <div style="margin-top:12px">
            <form method="get"><input type="hidden" name="page" value="attendance" /><div class="form-row"><input class="input" name="date" value="<?= h($_GET['date'] ?? date('Y-m-d')) ?>" placeholder="YYYY-MM-DD" /><button class="btn">Load</button></div></form>
            <?php
            $date = $_GET['date'] ?? date('Y-m-d');
            $path = sprintf('%s/attendance/%s/%s/%s/attendance-%s.ndjson', $ROOT_DB, substr($date,0,4), substr($date,5,2), substr($date,8,2), $date);
            if (!file_exists($path)) {
                echo "<div class='card small'>No attendance file for " . h($date) . "</div>";
            } else {
                $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                echo '<table class="table"><thead><tr><th>Time</th><th>Name</th><th>Admno</th><th>UID</th><th>Device</th><th>Result</th></tr></thead><tbody>';
                foreach ($lines as $l) {
                    $o = json_decode($l, true);
                    echo '<tr>';
                    echo '<td class="small">' . h($o['timestamp'] ?? '') . '</td>';
                    echo '<td>' . h($o['name'] ?? $o['user_id'] ?? '') . '</td>';
                    echo '<td class="small">' . h($o['admno_stored'] ?? $o['admno_supplied'] ?? '') . '</td>';
                    echo '<td class="small">' . h($o['uid'] ?? '') . '</td>';
                    echo '<td class="small">' . h($o['device_id'] ?? '') . '</td>';
                    echo '<td>' . h($o['result'] ?? '') . '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';
            }
            ?>
          </div>
        </div>

        <?php if (isset($_GET['action']) && $_GET['action'] == 'append_form'): ?>
          <div class="card">
            <div style="font-weight:700">Manual attendance</div>
            <form method="post">
              <input type="hidden" name="op" value="append_attendance">
              <div class="form-row"><input class="input" name="uid" placeholder="UID (card id)" required></div>
              <div class="form-row"><input class="input" name="admno" placeholder="Admission number (optional)"></div>
              <div class="form-row"><input class="input" name="device_id" placeholder="Device label (e.g. ESP32-GATE1)"></div>
              <div style="text-align:right"><button class="btn">Record</button></div>
            </form>
          </div>
        <?php endif; ?>

      <?php elseif ($page === 'attendance_by_user'): ?>
        <div class="card">
          <div style="display:flex;justify-content:space-between;align-items:center"><div style="font-weight:700">Attendance — by user</div><div class="small">Enter user id or choose</div></div>
          <div style="margin-top:12px">
            <form method="get"><input type="hidden" name="page" value="attendance_by_user" /><div class="form-row"><input class="input" name="user_id" placeholder="user-..." value="<?= h($_GET['user_id'] ?? '') ?>" /><button class="btn">Load</button></div></form>
            <?php if (isset($_GET['user_id'])): $u = read_json_file($ROOT_DB . '/users/user-' . $_GET['user_id'] . '.json'); if (!$u) { echo '<div class="small">User not found</div>'; } else { $path = $ROOT_DB . '/attendance_by_user/user-' . $u['user_id'] . '.ndjson'; if (!file_exists($path)) echo '<div class="small">No history</div>'; else { $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES); echo '<table class="table"><thead><tr><th>Time</th><th>Action</th><th>Device</th><th>Result</th></tr></thead><tbody>'; foreach ($lines as $l) { $o=json_decode($l,true); echo '<tr><td class="small">'.h($o['timestamp']??'').'</td><td>'.h($o['action']??'').'</td><td class="small">'.h($o['device_id']??'').'</td><td>'.h($o['result']??'').'</td></tr>'; } echo '</tbody></table>'; } } endif; ?>
          </div>
        </div>

      <?php elseif ($page === 'indexes'): ?>
        <div class="card">
          <div style="display:flex;justify-content:space-between;align-items:center"><div style="font-weight:700">Indexes</div><div><a class="btn" href="?page=indexes&action=regen">Regenerate indexes</a></div></div>
          <div style="margin-top:12px">
            <div class="small">names.json</div>
            <div class="code" style="margin-top:6px;max-height:160px;overflow:auto"><?= h(json_encode($indexes['names'], JSON_PRETTY_PRINT)) ?></div>
            <div class="small" style="margin-top:8px">admnos.json</div>
            <div class="code" style="margin-top:6px;max-height:160px;overflow:auto"><?= h(json_encode($indexes['admnos'], JSON_PRETTY_PRINT)) ?></div>
            <div class="small" style="margin-top:8px">uids.json</div>
            <div class="code" style="margin-top:6px;max-height:160px;overflow:auto"><?= h(json_encode($indexes['uids'], JSON_PRETTY_PRINT)) ?></div>
          </div>
        </div>
        <?php if (isset($_GET['action']) && $_GET['action'] === 'regen') { regenerate_indexes($ROOT_DB); header('Location: ?page=indexes'); exit; } ?>

      <?php elseif ($page === 'import_export'): ?>
        <div class="card">
          <div style="font-weight:700">Import users</div>
          <div class="small">Supported: JSON array, NDJSON (lines) — fields: admno, name, class, uids</div>
          <form method="post" enctype="multipart/form-data" style="margin-top:8px">
            <input type="hidden" name="op" value="import_users">
            <input type="file" name="import_file" required>
            <div style="text-align:right;margin-top:8px"><button class="btn">Upload & import</button></div>
          </form>
        </div>
        <div class="card">
          <div style="font-weight:700">Export</div>
          <div style="margin-top:8px"><a class="btn" href="?export=users&format=json">Users JSON</a> <a class="btn ghost" href="?export=users&format=ndjson">Users NDJSON</a> <a class="btn ghost" href="?export=users&format=csv">Users CSV</a></div>
        </div>

      <?php elseif ($page === 'backup'): ?>
        <div class="card">
          <div style="font-weight:700">Backup & Migrate</div>
          <div class="small">Create a zip of the entire /database directory.</div>
          <form method="post"><input type="hidden" name="op" value="backup_zip"><div style="text-align:right;margin-top:8px"><button class="btn">Create ZIP</button></div></form>
        </div>

      <?php elseif ($page === 'settings'): ?>
        <div class="card">
          <div style="font-weight:700">Settings</div>
          <div class="small" style="margin-top:8px">This UI operates directly on the on-disk database per your spec. Please ensure proper permissions and backups.</div>
          <div style="margin-top:10px"><strong>Guidance & recommended ops:</strong>
            <ul class="small">
              <li>Schedule daily backups (zip + rotate).</li>
              <li>For heavy load, migrate to a real DB (Postgres / MySQL) and keep this filesystem schema as archival export.</li>
              <li>Limit admin access to this UI and run over HTTPS only.</li>
            </ul>
          </div>
        </div>

      <?php else: ?>
        <div class="card">Unknown page</div>
      <?php endif; ?>

      <div class="footer">Titanium Attendance System — file-based DB viewer • Generated UI • <?= date('Y-m-d') ?></div>
    </div>
  </main>
</div>
</body>
</html>
<?php
// end of file
