<?php
/**
 * Titanium Attendance System - Professional Single File Admin UI (FIXED VERSION)
 * Filename: titanium-attendance-pro.php
 * Purpose: A complete overhaul to add professional features, enhanced analytics,
 * time-based sections (Today, Week, Month), a modern UI, and dynamic refresh capabilities.
 *
 * FIXES & ENHANCEMENTS:
 * 1. Robust directory creation and error checking.
 * 2. Simplified index regeneration logic.
 * 3. Consistent 12-Hour AM/PM formatting.
 * 4. AJAX loading for dashboard metrics on initial page load.
 * 5. Added a dedicated error message display for database root issues.
 *
 * Author: Generated for user (professional single-file deliverable)
 * Date: 2025-11-28
 */

// --------------------------------------------------
// CONFIG & CONSTANTS
// --------------------------------------------------
// Base database path, relative to this script.
$ROOT_DB = __DIR__ . DIRECTORY_SEPARATOR . 'database';
$DB_ERROR_MESSAGE = '';
$DEFAULT_LIMIT = 500; // Limit for stream reading and recent events

// --------------------------------------------------
// UTILITY FUNCTIONS
// --------------------------------------------------

function json_response($data) {
    header('Content-Type: application/json');
    echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

function ensure_dir($path) {
    global $DB_ERROR_MESSAGE;
    if (!is_dir($path)) {
        if (!mkdir($path, 0755, true)) {
            $DB_ERROR_MESSAGE = "Failed to create directory: " . htmlspecialchars($path) . ". Check filesystem permissions (should be writable by the web server).";
            return false;
        }
    }
    return true;
}

function normalize_uid($uid) {
    return strtoupper(preg_replace('/\s+/', '', (string)$uid));
}

function safe_write_json_atomic($path, $data) {
    $tmp = $path . '.tmp';
    $bytes = file_put_contents($tmp, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    if ($bytes === false) return false;
    // Check if the directory for the final file exists before renaming
    if (!ensure_dir(dirname($path))) return false;
    return rename($tmp, $path);
}

function append_ndjson_line($path, $obj) {
    if (!ensure_dir(dirname($path))) return false;
    $line = json_encode($obj, JSON_UNESCAPED_UNICODE) . "\n";
    return file_put_contents($path, $line, FILE_APPEND | LOCK_EX) !== false;
}

function read_json_file($path) {
    if (!file_exists($path)) return null;
    $txt = file_get_contents($path);
    if ($txt === false) return null;
    return json_decode($txt, true);
}

function list_users_dir($root) {
    $dir = $root . '/users';
    if (!ensure_dir($dir)) return [];
    $files = glob($dir . '/user-*.json');
    $out = [];
    foreach ($files as $f) {
        $data = json_decode(file_get_contents($f), true);
        if ($data) $out[$data['user_id']] = $data;
    }
    return $out;
}

function load_indexes($root) {
    $idx = $root . '/index';
    if (!ensure_dir($idx)) return ['names' => [], 'admnos' => [], 'uids' => []];
    $names = read_json_file($idx . '/names.json') ?? [];
    $admnos = read_json_file($idx . '/admnos.json') ?? [];
    $uids = read_json_file($idx . '/uids.json') ?? [];
    return ['names' => $names, 'admnos' => $admnos, 'uids' => $uids];
}

function regenerate_indexes($root) {
    $users = list_users_dir($root);
    $names = []; $admnos = []; $uids = [];
    foreach ($users as $u) {
        $user_id = $u['user_id'];
        $names[$user_id] = $u['name'];
        if (!empty($u['admno'])) $admnos[$u['admno']] = $user_id;
        if (!empty($u['uids']) && is_array($u['uids'])) {
            foreach ($u['uids'] as $card) {
                $norm = normalize_uid($card);
                if (!empty($norm)) $uids[$norm] = $user_id;
            }
        }
    }
    ensure_dir($root . '/index');
    safe_write_json_atomic($root . '/index/names.json', $names);
    safe_write_json_atomic($root . '/index/admnos.json', $admnos);
    safe_write_json_atomic($root . '/index/uids.json', $uids);
    return ['names' => $names, 'admnos' => $admnos, 'uids' => $uids];
}

function generate_id($prefix = 'id') {
    return $prefix . '_' . bin2hex(random_bytes(4));
}

function safe_append_attendance($root, $date, $event) {
    $parts = explode('-', $date);
    if (count($parts) !== 3) return false;
    list($y,$m,$d) = $parts;
    $dir = sprintf('%s/attendance/%s/%s/%s', $root, $y, $m, $d);
    $path = $dir . '/attendance-' . $date . '.ndjson';
    return append_ndjson_line($path, $event);
}

function safe_append_attendance_by_user($root, $user_id, $event) {
    $dir = $root . '/attendance_by_user';
    $path = $dir . '/user-' . $user_id . '.ndjson';
    return append_ndjson_line($path, $event);
}

/**
 * Professional Time Format: Converts ISO 8601 timestamp to 12-hour AM/PM format.
 * @param string $iso_time
 * @return string
 */
function format_timestamp_12h($iso_time) {
    try {
        $dt = new DateTime($iso_time);
        return $dt->format('M j Y, h:i:s A'); // e.g., Dec 31 2025, 04:30:15 PM
    } catch (Exception $e) {
        return $iso_time;
    }
}

/**
 * Calculates key metrics for a given set of attendance events.
 * @param array $events Array of attendance event objects.
 * @param array $users_map Map of user_id to user data.
 * @return array Metrics.
 */
function calculate_metrics($events, $users_map) {
    $total_scans = count($events);
    $accepted_events = 0;
    $rejected_events = 0;
    $users_attended = [];
    $event_types = ['present' => 0, 'in' => 0, 'out' => 0];

    foreach ($events as $event) {
        if (($event['result'] ?? '') === 'accepted') {
            $accepted_events++;
            $users_attended[$event['user_id']] = true;
            $action = strtolower($event['action'] ?? 'present');
            if (isset($event_types[$action])) {
                $event_types[$action]++;
            } else {
                $event_types['present']++; // Default to present if action is unknown
            }
        } else {
            $rejected_events++;
        }
    }

    $unique_users_attended = count($users_attended);
    $total_registered_users = count($users_map);

    $acceptance_rate = $total_scans > 0 ? ($accepted_events / $total_scans) * 100 : 0;
    $attendance_rate = $total_registered_users > 0 ? ($unique_users_attended / $total_registered_users) * 100 : 0;

    return [
        'total_scans' => $total_scans,
        'accepted_events' => $accepted_events,
        'rejected_events' => $rejected_events,
        'unique_attended' => $unique_users_attended,
        'total_registered' => $total_registered_users,
        'attendance_rate' => round($attendance_rate, 2),
        'acceptance_rate' => round($acceptance_rate, 2),
        'event_types' => $event_types
    ];
}


/**
 * Fetches all attendance records for a given date range.
 * @param string $start_date YYYY-MM-DD
 * @param string $end_date YYYY-MM-DD
 * @return array
 */
function get_attendance_in_range($root, $start_date, $end_date) {
    $all_events = [];
    try {
        $current = new DateTime($start_date);
        $end = new DateTime($end_date);
        $end->modify('+1 day'); // Ensure end date is inclusive

        while ($current < $end) {
            $date = $current->format('Y-m-d');
            $year = $current->format('Y');
            $month = $current->format('m');
            $day = $current->format('d');

            $path = sprintf('%s/attendance/%s/%s/%s/attendance-%s.ndjson', $root, $year, $month, $day, $date);

            if (file_exists($path)) {
                $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines !== false) {
                    foreach ($lines as $l) {
                        $o = json_decode($l, true);
                        if ($o) $all_events[] = $o;
                    }
                }
            }
            $current->modify('+1 day');
        }
    } catch (Exception $e) {
        // Handle invalid dates gracefully
    }

    // Sort events by timestamp descending
    usort($all_events, function($a, $b) {
        return strcmp($b['timestamp'] ?? '', $a['timestamp'] ?? '');
    });

    return $all_events;
}


// --------------------------------------------------
// DATA INITIALIZATION & ROUTING SETUP
// --------------------------------------------------
session_start();
$page = $_GET['page'] ?? 'dashboard';
$action = $_POST['action'] ?? $_GET['action'] ?? null;

// Ensure base database directory exists before proceeding
ensure_dir($ROOT_DB);

$users_map = list_users_dir($ROOT_DB);
$indexes = load_indexes($ROOT_DB);

// Define today, this week, and this month ranges
$dt_today = new DateTime();
$dt_week_start = (clone $dt_today)->modify('monday this week')->setTime(0, 0);
$dt_month_start = (clone $dt_today)->modify('first day of this month')->setTime(0, 0);

$today_date = $dt_today->format('Y-m-d');
$week_start_date = $dt_week_start->format('Y-m-d');
$month_start_date = $dt_month_start->format('Y-m-d');


// --- API / AJAX Endpoints ---
if (isset($_GET['api']) && $_GET['api'] === 'stats') {
    // Determine the date range for the dashboard stats
    $dashboard_view = $_GET['view'] ?? 'today';
    $start_date = $today_date;
    $end_date = $today_date;

    if ($dashboard_view === 'week') {
        $start_date = $week_start_date;
    } elseif ($dashboard_view === 'month') {
        $start_date = $month_start_date;
    }

    $events = get_attendance_in_range($ROOT_DB, $start_date, $end_date);
    $metrics = calculate_metrics($events, $users_map);

    // Return all stats and recent events
    json_response([
        'metrics' => $metrics,
        'users_count' => count($users_map),
        'uids_count' => count($indexes['uids']),
        'recent_events' => array_map(function($e) {
            $e['timestamp_formatted'] = format_timestamp_12h($e['timestamp']);
            return $e;
        }, array_slice($events, 0, 15)) // Limit recent events for dashboard
    ]);
}

// --- POST Actions (Attendance & User Management) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['op'])) {
    $op = $_POST['op'];

    if ($op === 'append_attendance') {
        $uid = normalize_uid($_POST['uid'] ?? '');
        $device = trim($_POST['device_id'] ?? 'MANUAL_ADMIN');
        $actionLabel = trim($_POST['action_label'] ?? 'present');
        $datetime_str = trim($_POST['datetime'] ?? 'now');

        try { $now = new DateTime($datetime_str); } catch (Exception $e) { $now = new DateTime(); }

        $user_id = $indexes['uids'][$uid] ?? null;
        $result = 'rejected'; $msg = 'UID not assigned or invalid.';
        $u = null;

        if ($user_id) {
            $u = read_json_file($ROOT_DB . '/users/user-' . $user_id . '.json');
            if ($u) {
                // Check for duplicate attendance today (simple rule: one accepted entry per user per day)
                $date = $now->format('Y-m-d');
                $dayPath = sprintf('%s/attendance/%s/%s/%s/attendance-%s.ndjson', $ROOT_DB, substr($date,0,4), substr($date,5,2), substr($date,8,2), $date);
                $already_accepted = false;
                if (file_exists($dayPath)) {
                    $lines = file($dayPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                    if ($lines !== false) {
                        foreach ($lines as $l) {
                            $o = json_decode($l, true);
                            if ($o && ($o['user_id'] ?? '') === $user_id && ($o['result'] ?? '') === 'accepted') {
                                $already_accepted = true;
                                break;
                            }
                        }
                    }
                }

                if ($already_accepted) {
                    $result = 'rejected';
                    $msg = 'User attendance already recorded today.';
                } else {
                    $result = 'accepted';
                    $msg = 'Attendance successfully recorded.';
                }
            }
        }

        $event = [
            'event_id' => generate_id('evt'),
            'timestamp' => $now->format(DATE_ATOM),
            'user_id' => $user_id,
            'admno_stored' => $u['admno'] ?? null,
            'name' => $u['name'] ?? null,
            'class' => $u['class'] ?? null,
            'uid' => $uid,
            'device_id' => $device,
            'action' => $actionLabel,
            'result' => $result,
            'message' => $msg
        ];

        $date = $now->format('Y-m-d');
        safe_append_attendance($ROOT_DB, $date, $event);
        if ($user_id) safe_append_attendance_by_user($ROOT_DB, $user_id, $event);

        if ($result === 'accepted') {
            $_SESSION['flash_success'] = "Attendance recorded for **" . h($u['name'] ?? 'UID ' . $uid) . "** successfully.";
        } else {
            $_SESSION['flash_error'] = "Attendance rejected for UID **" . h($uid) . "**: " . h($msg);
        }
        header('Location: ?page=dashboard&view=today'); exit;
    }

    if ($op === 'create_user') {
        ensure_dir($ROOT_DB . '/users');
        $user_id = generate_id('usr');
        $uids_raw = array_filter(preg_split('/[\r\n,;]+/', $_POST['uids'] ?? ''));
        $uids_normalized = array_map('normalize_uid', $uids_raw);
        $record = [
            'user_id' => $user_id,
            'admno' => trim($_POST['admno']),
            'name' => trim($_POST['name']),
            'class' => trim($_POST['class']),
            'uids' => $uids_normalized,
            'created_at' => (new DateTime())->format(DATE_ATOM),
            'updated_at' => (new DateTime())->format(DATE_ATOM)
        ];
        if (safe_write_json_atomic($ROOT_DB . '/users/user-' . $user_id . '.json', $record)) {
            regenerate_indexes($ROOT_DB);
            $_SESSION['flash_success'] = "User **" . h($record['name']) . "** created.";
            header('Location: ?page=users&action=view&user_id=' . $user_id); exit;
        } else {
            $_SESSION['flash_error'] = "Failed to create user file.";
            header('Location: ?page=users&action=create'); exit;
        }
    }
    if ($op === 'edit_user') {
         $user_id = $_POST['user_id'] ?? '';
         $path = $ROOT_DB . '/users/user-' . $user_id . '.json';
         $data = read_json_file($path);
         if ($data) {
            $uids_raw = array_filter(preg_split('/[\r\n,;]+/', $_POST['uids'] ?? ''));
            $data['name'] = trim($_POST['name']);
            $data['class'] = trim($_POST['class']);
            $data['uids'] = array_map('normalize_uid', $uids_raw);
            $data['updated_at'] = (new DateTime())->format(DATE_ATOM);
            if (safe_write_json_atomic($path, $data)) {
                regenerate_indexes($ROOT_DB);
                $_SESSION['flash_success'] = "User **" . h($data['name']) . "** updated.";
                header('Location: ?page=users&action=view&user_id=' . $user_id); exit;
            } else {
                $_SESSION['flash_error'] = "Failed to update user file.";
                header('Location: ?page=users&action=edit&user_id=' . $user_id); exit;
            }
         }
    }
    // Added Index Regeneration after user actions
    if ($op === 'regen_indexes') {
         regenerate_indexes($ROOT_DB);
         $_SESSION['flash_success'] = 'Indexes regenerated successfully.';
         header('Location: ?page=indexes'); exit;
    }
}

// --- Flash Messages ---
$flash_ok = $_SESSION['flash_success'] ?? null; unset($_SESSION['flash_success']);
$flash_err = $_SESSION['flash_error'] ?? null; unset($_SESSION['flash_error']);
$flash_warn = $_SESSION['flash_warn'] ?? null; unset($_SESSION['flash_warn']);

// --- HTML Helper ---
function h($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

?><!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1.0">
<title>Titanium Pro Attendance — Admin Console</title>
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;600;700;800&display=swap" rel="stylesheet">
<style>
:root{--bg:#0f1724;--card:#1a2536;--accent:#0ea5e9;--muted:#94a3b8;--glass:rgba(255,255,255,0.04);--success:#10b981;--error:#ef4444;--warning:#f59e0b;--shadow:0 10px 30px rgba(0,0,0,0.4)}
*{box-sizing:border-box;font-family:'Inter',ui-sans-serif,system-ui,Segoe UI,Roboto,'Helvetica Neue',Arial}
body{margin:0;background:linear-gradient(180deg,var(--bg),#071026);color:#e6eef6;line-height:1.6}
.app{display:grid;grid-template-columns:260px 1fr;min-height:100vh}
@media (max-width: 1024px) { .app{grid-template-columns:1fr;} .sidebar{position:fixed;top:0;left:0;height:100%;z-index:100;background:var(--bg);padding:20px;box-shadow:var(--shadow);transform:translateX(-100%);transition:transform 0.3s ease-out;} .sidebar.open{transform:translateX(0);} .main-content{margin-left:0; padding-top: 60px;} .menu-button{display:block!important;} }
.sidebar{padding:18px;border-right:1px solid rgba(255,255,255,0.03);background:linear-gradient(180deg,rgba(255,255,255,0.02),transparent)}
.main-content{padding-top:0}
.brand{font-size:20px;font-weight:800;display:flex;align-items:center;gap:12px;color:var(--accent)}
.logo{width:48px;height:48px;border-radius:12px;background:linear-gradient(135deg,var(--accent),#0369a1);display:flex;align-items:center;justify-content:center;color:white;font-size:24px;box-shadow: 0 4px 8px rgba(0,0,0,0.3)}
.nav{margin-top:20px}
.nav a{display:flex;align-items:center;gap:12px;padding:12px 14px;border-radius:10px;color:var(--muted);text-decoration:none;margin-bottom:6px;transition: background 0.2s, color 0.2s, transform 0.1s}
.nav a:hover{background:rgba(14,165,233,0.1);color:#fff;transform:translateX(2px)}
.nav a.active{background:rgba(14,165,233,0.2);color:white;font-weight:600;border-left:4px solid var(--accent);padding-left:10px}
.header{display:flex;align-items:center;justify-content:space-between;padding:18px 24px;border-bottom:1px solid rgba(255,255,255,0.05);backdrop-filter: blur(8px);position:sticky;top:0;z-index:10;background:rgba(15,23,36,0.95)}
.menu-button{display:none;font-size:24px;background:none;border:none;color:white;cursor:pointer;padding:0}
.container{padding:24px}
.card{background:var(--card);border-radius:16px;padding:20px;margin-bottom:24px;box-shadow:var(--shadow);border: 1px solid rgba(255,255,255,0.05);transition: transform 0.3s ease-out, box-shadow 0.3s ease-out}
.card:hover{transform: translateY(-2px);box-shadow: 0 15px 40px rgba(0,0,0,0.5)}
.kpi{display:grid;grid-template-columns:repeat(auto-fit, minmax(180px, 1fr));gap:16px;margin-top:20px}
.kpi .item{padding:18px;border-radius:12px;background:linear-gradient(180deg,rgba(255,255,255,0.02),transparent);text-align:left;border:1px solid rgba(255,255,255,0.08);box-shadow: 0 2px 10px rgba(0,0,0,0.2)}
.kpi-title{font-size:14px;color:var(--muted);font-weight:600;text-transform:uppercase}
.kpi-value{font-size:28px;font-weight:800;color:#fff;margin-top:4px;display:flex;align-items:center;gap:8px}
.kpi-value .icon{color:var(--accent);font-size:20px}
.kpi-pct{font-size:18px;font-weight:700;color:var(--success)}
.kpi .error-text {color: var(--error); font-weight: 700;}
.small{font-size:13px;color:var(--muted)}
.table{width:100%;border-collapse:separate;border-spacing:0;margin-top:15px}
.table th,.table td{padding:12px;text-align:left;border-bottom:1px solid rgba(255,255,255,0.05)}
.table th{font-weight:700;color:#c8d4e4;text-transform:uppercase;font-size:13px;background:rgba(255,255,255,0.02)}
.table tr:last-child td{border-bottom:none}
.btn{display:inline-block;padding:10px 16px;border-radius:10px;background:var(--accent);color:#0e131d;text-decoration:none;font-weight:700;border:none;cursor:pointer;transition:background 0.3s, transform 0.1s;box-shadow: 0 4px 8px rgba(14,165,233,0.3)}
.btn:hover{background:#0369a1;transform:translateY(-1px)}
.btn.ghost{background:transparent;border:1px solid rgba(255,255,255,0.15);color:var(--accent);font-weight:600;box-shadow:none;padding:8px 14px}
.btn.ghost:hover{background:rgba(255,255,255,0.05);border-color:var(--accent);color:#fff}
.form-row{display:flex;gap:12px;margin-bottom:12px}
.form-row > label {flex-grow: 1;}
.input,textarea,select{background:rgba(0,0,0,0.2);border:1px solid rgba(255,255,255,0.15);padding:12px;border-radius:10px;color:inherit;width:100%;transition:border-color 0.2s, box-shadow 0.2s}
.input:focus,textarea:focus,select:focus{border-color:var(--accent);box-shadow: 0 0 0 3px rgba(14,165,233,0.3);outline:none}
.code{font-family:monospace;background:rgba(0,0,0,0.3);padding:10px;border-radius:8px;color:#9fe3e3;white-space:pre-wrap;overflow-x:auto}
.badge{display:inline-block;padding:6px 10px;border-radius:999px;font-size:12px;font-weight:600;text-transform:uppercase}
.badge.success{background:rgba(16,185,129,0.2);color:var(--success)}
.badge.error{background:rgba(239,68,68,0.2);color:var(--error)}
.badge.neutral{background:rgba(255,255,255,0.1);color:var(--muted)}
.animate-fade{animation:fadeIn .5s ease both}
@keyframes fadeIn { from {opacity:0; transform: translateY(10px)} to {opacity:1; transform:none} }
.footer{padding:12px 24px;color:var(--muted);font-size:12px;text-align:center}
.tab-nav{display:flex;border-bottom:1px solid rgba(255,255,255,0.1)}
.tab-nav a{padding:10px 18px;text-decoration:none;color:var(--muted);font-weight:600;transition:color 0.2s, border-bottom 0.2s}
.tab-nav a.active{color:white;border-bottom:3px solid var(--accent);margin-bottom:-1px}
.chart-container{height:250px;background:rgba(0,0,0,0.1);border-radius:8px;display:flex;align-items:center;justify-content:center;color:var(--muted);font-size:14px;border:1px dashed rgba(255,255,255,0.1)}
.alert{padding:15px;border-radius:10px;margin-bottom:15px;font-weight:600}
.alert-success{border:1px solid var(--success);background:rgba(16,185,129,0.1);color:var(--success)}
.alert-error{border:1px solid var(--error);background:rgba(239,68,68,0.1);color:var(--error)}
.alert-warning{border:1px solid var(--warning);background:rgba(245,158,11,0.1);color:var(--warning)}
</style>
<script>
// Icon mapping (Unicode Emojis)
const Icons = {
    'total_scans': '⚡', 'accepted_events': '✅', 'rejected_events': '❌',
    'unique_attended': '🧑‍🎓', 'total_registered': '👥',
    'attendance_rate': '🎯', 'acceptance_rate': '👍',
    'users': '🧑‍🎓', 'uids': '💳', 'refresh': '🔄', 'view': '👁️',
    'present': '🟢', 'in': '⬆️', 'out': '⬇️', 'leave': '🟡',
    'menu': '☰'
};

function formatPercentage(value) {
    if (isNaN(value)) return '0%';
    return value.toFixed(2) + '%';
}

function updateEventSummary(metrics) {
    const summaryContainer = document.getElementById('today-event-summary');
    if (!summaryContainer) return;

    if (metrics.total_scans === 0) {
        summaryContainer.innerHTML = '<li>No events recorded yet.</li>';
        return;
    }

    let html = '';
    const eventMap = {
        'present': 'Present/Generic',
        'in': 'Clock In',
        'out': 'Clock Out',
        'leave': 'Leave Recorded'
    };

    for (const key in eventMap) {
        const count = metrics.event_types[key] || 0;
        const icon = Icons[key] || '•';
        const percent = metrics.accepted_events > 0 ? (count / metrics.accepted_events * 100).toFixed(1) : 0;
        html += `
            <li style="margin-bottom: 5px; display: flex; justify-content: space-between; align-items: center; padding: 4px 0; border-bottom: 1px dashed rgba(255,255,255,0.05);">
                <span style="font-weight: 500; color: #fff;">${icon} ${eventMap[key]}</span>
                <span style="font-size: 14px; font-weight: 700; color: var(--accent);">${count} (${percent}%)</span>
            </li>
        `;
    }
    summaryContainer.innerHTML = html;
}

function refreshDashboardData(view, btn) {
    const metricsContainer = document.getElementById('dashboard-metrics');
    const recentContainer = document.getElementById('recent-events-table');
    const originalBtnText = btn.innerHTML;
    btn.innerHTML = `${Icons.refresh} Reloading...`;
    btn.disabled = true;

    // Show loading placeholders
    if (metricsContainer) metricsContainer.innerHTML = `<div class="kpi-value" style="text-align:center; grid-column: 1 / -1; color: var(--accent); margin-top: 10px;">... Loading Metrics ...</div>`;
    if (recentContainer) recentContainer.innerHTML = `<tr><td colspan="6" class="small" style="text-align:center; padding: 20px;">Fetching recent logs...</td></tr>`;
    updateEventSummary({total_scans:0, accepted_events:0, event_types: {}}); // Clear summary

    fetch(`?api=stats&view=${view}`)
        .then(response => response.json())
        .then(data => {
            const m = data.metrics;

            // Update Metrics (KPIs)
            if (metricsContainer) {
                metricsContainer.innerHTML = `
                    <div class="item animate-fade">
                        <div class="kpi-title">Total Scans</div>
                        <div class="kpi-value"><span class="icon">${Icons.total_scans}</span>${m.total_scans}</div>
                        <div class="small">Accepted: ${m.accepted_events} | Rejected: ${m.rejected_events}</div>
                    </div>
                    <div class="item animate-fade">
                        <div class="kpi-title">Attendance Rate</div>
                        <div class="kpi-value"><span class="icon">${Icons.attendance_rate}</span><span class="kpi-pct" style="color:${m.attendance_rate < 50 ? 'var(--error)' : 'var(--success)'}">${formatPercentage(m.attendance_rate)}</span></div>
                        <div class="small">Users Attended: ${m.unique_attended} / ${m.total_registered}</div>
                    </div>
                    <div class="item animate-fade">
                        <div class="kpi-title">Scan Acceptance</div>
                        <div class="kpi-value"><span class="icon">${Icons.acceptance_rate}</span><span class="kpi-pct" style="color:${m.acceptance_rate < 80 ? 'var(--warning)' : 'var(--success)'}">${formatPercentage(m.acceptance_rate)}</span></div>
                        <div class="small">Success rate for all scans.</div>
                    </div>
                    <div class="item animate-fade">
                        <div class="kpi-title">Total Registered Users</div>
                        <div class="kpi-value"><span class="icon">${Icons.users}</span>${data.users_count}</div>
                        <div class="small">Indexed UIDs: ${data.uids_count}</div>
                    </div>
                `;
            }

            // Update Event Summary (only relevant for Today view, but run for all)
            updateEventSummary(m);


            // Update Recent Events Table
            if (recentContainer) {
                if (data.recent_events.length === 0) {
                    recentContainer.innerHTML = `<tr><td colspan="6" class="small" style="text-align:center; padding: 20px;">No events recorded in this period.</td></tr>`;
                } else {
                    recentContainer.innerHTML = data.recent_events.map(r => {
                        const isAccepted = r.result === 'accepted';
                        const badgeClass = isAccepted ? 'success' : 'error';
                        return `
                            <tr>
                                <td class="small">${r.timestamp_formatted}</td>
                                <td><strong>${r.name || 'N/A'}</strong></td>
                                <td class="small">${r.admno_stored || 'N/A'}</td>
                                <td class="small">${r.uid || 'N/A'}</td>
                                <td class="small">${r.device_id || 'N/A'}</td>
                                <td><span class="badge ${badgeClass}">${r.result.toUpperCase()}</span></td>
                            </tr>
                        `;
                    }).join('');
                }
            }

            btn.innerHTML = `${Icons.refresh} Data Refreshed`;
            setTimeout(() => { btn.innerHTML = originalBtnText; btn.disabled = false; }, 1500);

        })
        .catch(error => {
            console.error('Error fetching stats:', error);
            if (metricsContainer) metricsContainer.innerHTML = `<div class="kpi-value error-text" style="grid-column: 1 / -1; text-align:center">❌ Error loading data. See console.</div>`;
            btn.innerHTML = `${Icons.error} Reload Error`;
            setTimeout(() => { btn.innerHTML = originalBtnText; btn.disabled = false; }, 3000);
        });
}

document.addEventListener('DOMContentLoaded', () => {
    const currentView = '<?= $_GET['view'] ?? 'today' ?>';
    const currentPage = '<?= $page ?>';
    const refreshBtn = document.getElementById('header-refresh-btn');
    const menuBtn = document.getElementById('menu-toggle-btn');
    const sidebar = document.getElementById('sidebar-menu');

    // Mobile Menu Toggle
    if (menuBtn && sidebar) {
        menuBtn.addEventListener('click', () => {
            sidebar.classList.toggle('open');
        });
        // Close sidebar when a navigation link is clicked
        sidebar.querySelectorAll('a').forEach(link => {
            link.addEventListener('click', () => {
                sidebar.classList.remove('open');
            });
        });
    }

    if (currentPage === 'dashboard' && refreshBtn) {
        // Initial load of dashboard data (mimic a refresh call)
        refreshDashboardData(currentView, refreshBtn);
        refreshBtn.addEventListener('click', (e) => {
            e.preventDefault();
            refreshDashboardData(currentView, refreshBtn);
        });
    } else if (refreshBtn) {
        // For other pages, full reload is simpler
        refreshBtn.addEventListener('click', (e) => {
            e.preventDefault();
            window.location.reload();
        });
    }

    // Confirmation for destructive actions
    window.confirmAction = function(message) {
        // Check if running in a Canvas environment (where native confirm is blocked)
        if (typeof confirm !== 'undefined') {
            return confirm(message);
        } else {
            // Fallback for environments where confirm is disabled
            console.warn("Native confirm dialog blocked. Destructive action prevented.");
            alert("Action confirmation required: " + message);
            return false;
        }
    }
});
</script>
</head>
<body>
<div class="app">
  <aside class="sidebar" id="sidebar-menu">
    <div class="brand"><div class="logo">T</div> Titanium Pro</div>
    <div class="small" style="margin-top:6px">Attendance & DB Admin Console</div>
    <nav class="nav">
      <?php
      $navs = [
          'dashboard' => 'Dashboard',
          'analytics' => 'Analytics & Graphs',
          'users' => 'Users Management',
          'attendance' => 'Attendance By Day',
          'indexes' => 'Indexes',
          'import_export' => 'Import / Export',
          'backup' => 'Backup / Migrate',
      ];
      foreach ($navs as $k=>$label) {
          $cls = ($page === $k) ? 'active' : '';
          // Using simple emojis for icons
          $icon_map = ['dashboard'=>'🏠', 'analytics'=>'📈', 'users'=>'🧑‍🎓', 'attendance'=>'📅', 'indexes'=>'🗃️', 'import_export'=>'📤', 'backup'=>'💾'];
          $icon = $icon_map[$k] ?? '•';
          echo "<a class='$cls' href='?page=$k'>{$icon} " . h($label) . "</a>";
      }
      ?>
    </nav>
    <div style="margin-top:20px" class="small">Quick Actions</div>
    <div style="margin-top:8px; display:flex; gap: 8px;">
      <a class="btn" href="?page=users&action=create">New User</a>
      <a class="btn ghost" href="?page=import_export">Import Data</a>
    </div>
    <div style="height:30px"></div>
    <div class="small">DB Root Path</div>
    <div class="code" style="margin-top:6px;word-break:break-all;">/<?= h(trim(str_replace(__DIR__,'','' . $ROOT_DB), '/')) ?></div>
  </aside>
  <main class="main-content">
    <div class="header">
      <button class="menu-button" id="menu-toggle-btn"><?= Icons['menu'] ?></button>
      <div>
        <div style="font-weight:700; font-size: 20px"><?php
            $title = ['dashboard'=>'Dashboard Overview', 'analytics'=>'System Analytics', 'users'=>'Users Management', 'attendance'=>'Attendance Log', 'indexes'=>'Database Indexes', 'import_export'=>'Data Import / Export', 'backup'=>'Backup & Migration'];
            echo $title[$page] ?? 'System Console';
        ?></div>
        <div class="small">Professional Interface for file-based attendance system</div>
      </div>
      <div style="display:flex;gap:12px;align-items:center">
        <button id="header-refresh-btn" class="btn ghost"><?= Icons['refresh'] ?> Refresh Data</button>
      </div>
    </div>
    <div class="container">
      <?php if ($DB_ERROR_MESSAGE): ?><div class="alert alert-error animate-fade"><strong>❌ Database Error:</strong> <?= $DB_ERROR_MESSAGE ?></div><?php endif; ?>
      <?php if ($flash_ok): ?><div class="alert alert-success animate-fade"><strong>✅ Success:</strong> <?= h($flash_ok) ?></div><?php endif; ?>
      <?php if ($flash_err): ?><div class="alert alert-error animate-fade"><strong>❌ Error:</strong> <?= h($flash_err) ?></div><?php endif; ?>
      <?php if ($flash_warn): ?><div class="alert alert-warning animate-fade"><strong>⚠️ Warning:</strong> <?= h($flash_warn) ?></div><?php endif; ?>

      <?php if ($page === 'dashboard'): ?>
        <?php $view = $_GET['view'] ?? 'today'; ?>
        <div class="card animate-fade" style="padding: 12px 20px 0 20px;">
          <div class="tab-nav">
            <a class="<?= $view === 'today' ? 'active' : '' ?>" href="?page=dashboard&view=today">Today (<?= $dt_today->format('M j') ?>)</a>
            <a class="<?= $view === 'week' ? 'active' : '' ?>" href="?page=dashboard&view=week">This Week (<?= $dt_week_start->format('M j') ?> - <?= $dt_today->format('M j') ?>)</a>
            <a class="<?= $view === 'month' ? 'active' : '' ?>" href="?page=dashboard&view=month">This Month (<?= $dt_month_start->format('M Y') ?>)</a>
          </div>
        </div>

        <div class="card animate-fade">
            <div style="font-size:18px;font-weight:700">Key Performance Indicators (KPIs)</div>
            <div class="kpi" id="dashboard-metrics">
                <div class="kpi-value" style="text-align:center; grid-column: 1 / -1; color: var(--accent); margin-top: 10px;">... Initializing Dashboard Data ...</div>
            </div>
        </div>

        <?php if ($view === 'today'): ?>
            <div class="card animate-fade" style="display:grid; grid-template-columns: 2fr 1fr; gap: 20px">
                <div>
                    <div style="font-weight:700; font-size: 18px">Quick Manual Attendance Log</div>
                    <div class="small" style="margin-bottom:15px">Record an event using a UID/Card ID. This is a manual override.</div>
                    <form method="post">
                        <input type="hidden" name="op" value="append_attendance">
                        <div class="form-row">
                            <label><div class="small">UID (Card ID)</div><input class="input" name="uid" placeholder="e.g., 0A1B2C3D" required></label>
                            <label><div class="small">Date & Time (24-hour input)</div><input class="input" type="datetime-local" name="datetime" value="<?= (new DateTime())->format('Y-m-d\TH:i') ?>" required></label>
                        </div>
                        <div class="form-row">
                            <label><div class="small">Device/Source Label</div><input class="input" name="device_id" placeholder="e.g. ADMIN_OVERRIDE" value="ADMIN_OVERRIDE"></label>
                            <label><div class="small">Action Type</div>
                                <select class="input" name="action_label">
                                    <option value="present"><?= Icons['present'] ?> Present (Default)</option>
                                    <option value="in"><?= Icons['in'] ?> Clock In</option>
                                    <option value="out"><?= Icons['out'] ?> Clock Out</option>
                                    <option value="leave"><?= Icons['leave'] ?> On Leave</option>
                                </select>
                            </label>
                        </div>
                        <div style="text-align:right"><button class="btn">Record Attendance</button></div>
                    </form>
                </div>
                <div class="card" style="margin:0; background:rgba(0,0,0,0.2); border:1px solid rgba(255,255,255,0.1)">
                    <div style="font-weight:700">Attendance Summary (Today)</div>
                    <div class="small">Detailed breakdown of accepted event types:</div>
                    <ul style="list-style:none; padding:0; margin-top:10px" id="today-event-summary">
                        <li>... Loading summary ...</li>
                    </ul>
                </div>
            </div>
        <?php endif; // End Today view specific features ?>

        <div class="card animate-fade">
            <div style="font-weight:700; font-size: 18px">Recent Attendance Log (Top 15 Events)</div>
            <table class="table">
                <thead><tr><th>Time (12h)</th><th>Name</th><th>Admno</th><th>UID</th><th>Device</th><th>Result</th></tr></thead>
                <tbody id="recent-events-table">
                    <tr><td colspan="6" class="small" style="text-align:center; padding: 20px;">Fetching recent logs...</td></tr>
                </tbody>
            </table>
        </div>

      <?php elseif ($page === 'analytics'): ?>
        <div class="card animate-fade">
            <div style="font-weight:700; font-size: 18px">Attendance Percentage & Trend Analysis [Image of attendance trend chart]</div>
            <div class="small" style="margin-bottom:20px">Visualizations illustrating attendance trends, acceptance rates, and user performance over time.</div>

            <div style="display:grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px">
                <div class="card" style="margin:0; background:rgba(0,0,0,0.1)">
                    <div style="font-weight:600">Monthly Attendance Trends</div>
                    <div class="chart-container">Placeholder for Line Chart (Shows attendance rate over the last 12 months)</div>
                </div>
                <div class="card" style="margin:0; background:rgba(0,0,0,0.1)">
                    <div style="font-weight:600">Top 10 Acceptance Rate by Device</div>
                    <div class="chart-container">Placeholder for Bar Chart (Shows successful scan rate per device ID)</div>
                </div>
                <div class="card" style="margin:0; background:rgba(0,0,0,0.1); grid-column: 1 / -1;">
                    <div style="font-weight:600">User Performance Quartiles</div>
                    <div class="chart-container" style="height:150px">Placeholder for User Distribution Analysis (e.g., Percentage of users above 90% attendance)</div>
                </div>
            </div>
            <div class="alert alert-warning" style="margin-top:20px">
                **Note:** Real chart libraries (like Chart.js or D3.js) are not included to maintain the single-file nature, but the data analysis framework is available.
            </div>
        </div>


      <?php elseif ($page === 'users'): ?>
        <?php if (isset($_GET['action']) && $_GET['action'] === 'view' && isset($_GET['user_id'])): $uid = $_GET['user_id']; $u = read_json_file($ROOT_DB . '/users/user-' . $uid . '.json'); if ($u): ?>
          <div class="card animate-fade">
            <div style="display:flex;justify-content:space-between;align-items:center"><div style="font-size:22px;font-weight:700">Student Full Details: <?= h($u['name']) ?></div> <div><a class="btn ghost" href="?page=users">← Back to list</a> <a class="btn" href="?page=users&action=edit&user_id=<?= h($u['user_id']) ?>">Edit Record</a></div></div>
            <div style="margin-top:20px; display:grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px;">
                <!-- Left Column: Primary Details -->
                <div class="card" style="margin:0; background:rgba(0,0,0,0.2)">
                    <div style="font-weight:700; margin-bottom:10px">User Profile</div>
                    <table class="table" style="width:100%">
                        <tr><th style="width:150px">Name</th><td><strong><?= h($u['name']) ?></strong></td></tr>
                        <tr><th>Admission No.</th><td><span class="badge success"><?= h($u['admno']) ?></span></td></tr>
                        <tr><th>Class/Section</th><td><?= h($u['class']) ?></td></tr>
                        <tr><th>User ID (System)</th><td><span class="code" style="font-size:12px;"><?= h($u['user_id']) ?></span></td></tr>
                        <tr><th>Record Created</th><td><?= h(format_timestamp_12h($u['created_at'])) ?></td></tr>
                        <tr><th>Last Updated</th><td><?= h(format_timestamp_12h($u['updated_at'])) ?></td></td></tr>
                    </table>
                </div>

                <!-- Right Column: UID & History Link -->
                <div class="card" style="margin:0; background:rgba(0,0,0,0.2)">
                    <div style="font-weight:700; margin-bottom:10px">Assigned Authentication IDs</div>
                    <div class="small">The following UIDs are linked to this student for attendance tracking.</div>
                    <div class="code" style="margin-top:10px; max-height:200px; overflow-y:auto;"><?= h(implode("\n", $u['uids'])) ?></div>
                    <div style="margin-top:20px">
                        <a class="btn" href="?page=attendance_by_user&user_id=<?= h($u['user_id']) ?>">View Full Attendance History (Future Feature)</a>
                    </div>
                </div>
            </div>
          </div>
        <?php else: ?>
             <div class="alert alert-error animate-fade">User ID not found.</div>
        <?php endif; ?>
        <?php elseif (isset($_GET['action']) && $_GET['action'] === 'create'): ?>
          <div class="card animate-fade">
            <div style="font-weight:700; font-size: 18px">Create New Student/User Record</div>
            <form method="post">
              <input type="hidden" name="op" value="create_user">
              <div class="form-row"><label><div class="small">Full Name</div><input class="input" name="name" placeholder="Full Name" required></label></div>
              <div class="form-row"><label><div class="small">Admission Number (Unique)</div><input class="input" name="admno" placeholder="Admission Number (unique identifier)" required></label></div>
              <div class="form-row"><label><div class="small">Class / Section</div><input class="input" name="class" placeholder="Class / Section (e.g., 10th Grade A)"></label></div>
              <div class="form-row"><label><div class="small">Card UIDs (One per line or comma separated)</div><textarea class="input" name="uids" rows="4" placeholder="Card UIDs (comma or newline separated. Must be normalized)"></textarea></label></div>
              <div style="text-align:right"><button class="btn">Create User</button></div>
            </form>
          </div>
        <?php elseif (isset($_GET['action']) && $_GET['action'] === 'edit' && isset($_GET['user_id'])): $uid = $_GET['user_id']; $u = read_json_file($ROOT_DB . '/users/user-' . $uid . '.json'); if ($u): ?>
          <div class="card animate-fade">
            <div style="font-weight:700; font-size: 18px">Edit Student Record — <?= h($u['name']) ?></div>
            <form method="post">
              <input type="hidden" name="op" value="edit_user">
              <input type="hidden" name="user_id" value="<?= h($u['user_id']) ?>">
              <div class="form-row"><label><div class="small">Full Name</div><input class="input" name="name" value="<?= h($u['name']) ?>"></label></div>
              <div class="form-row"><label><div class="small">Class / Section</div><input class="input" name="class" value="<?= h($u['class']) ?>"></label></div>
              <div class="form-row"><label><div class="small">Card UIDs (One per line or comma separated)</div><textarea class="input" name="uids" rows="4" placeholder="Update UIDs. One per line or comma-separated."><?= h(implode("\n", $u['uids'])) ?></textarea></label></div>
              <div style="text-align:right"><button class="btn">Save Changes</button></div>
            </form>
          </div>
        <?php else: ?>
             <div class="alert alert-error animate-fade">User ID not found.</div>
        <?php endif; ?>

        <?php if (!isset($_GET['action']) || in_array($_GET['action'] ?? '', ['view', 'edit', 'create'])): ?>
        <div class="card animate-fade">
          <div style="display:flex;justify-content:space-between;align-items:center">
            <div style="font-weight:700; font-size: 18px">Registered Students (Total: <?= count($users_map) ?>)</div>
            <div><a class="btn" href="?page=users&action=create">New Student</a> <a class="btn ghost" href="?export=users&format=csv">Export CSV</a></div>
          </div>
          <div style="margin-top:12px">
            <form method="get"><input type="hidden" name="page" value="users" /><div class="form-row"><input class="input" name="q" placeholder="Search name / admno / uid" value="<?= h($_GET['q'] ?? '') ?>" style="max-width:300px" /><button class="btn ghost"><?= Icons['refresh'] ?> Search</button></div></form>
            <table class="table">
              <thead><tr><th>Name</th><th>Admno</th><th>Class</th><th>UIDs</th><th>Actions</th></tr></thead>
              <tbody>
              <?php
              $q = trim($_GET['q'] ?? '');
              $display_count = 0;
              foreach ($users_map as $u) {
                  if ($q !== '') {
                      $ok = stripos($u['name'],$q) !== false || stripos($u['admno'],$q)!==false;
                      foreach ($u['uids'] as $uu) if ($ok===false && stripos($uu,$q)!==false) $ok=true;
                      if (!$ok) continue;
                  }
                  $display_count++;
                  echo '<tr>';
                  echo '<td><strong>' . h($u['name']) . '</strong></td>';
                  echo '<td class="small">' . h($u['admno']) . '</td>';
                  echo '<td class="small">' . h($u['class']) . '</td>';
                  echo '<td class="small">' . h(implode(', ', array_slice($u['uids'], 0, 2))) . (count($u['uids']) > 2 ? '...' : '') . '</td>';
                  echo '<td><a class="btn ghost" href="?page=users&action=view&user_id=' . h($u['user_id']) . '">'.Icons['view'].' View Full</a></td>';
                  echo '</tr>';
                  if ($display_count > 200) { echo '<tr><td colspan="5" class="small" style="text-align:center">Display limit reached. Refine search.</td></tr>'; break; }
              }
              ?>
              </tbody>
            </table>
          </div>
        </div>
        <?php endif; ?>

      <?php elseif ($page === 'attendance'): ?>
        <div class="card animate-fade">
          <div style="font-weight:700; font-size: 18px">Attendance Log by Date</div>
          <div class="small" style="margin-bottom:15px">Select a date to view all recorded attendance events for that day.</div>
          <div style="margin-top:12px">
            <form method="get"><input type="hidden" name="page" value="attendance" /><div class="form-row"><input class="input" type="date" name="date" value="<?= h($_GET['date'] ?? $today_date) ?>" required placeholder="YYYY-MM-DD" style="max-width:300px;" /><button class="btn">Load Day</button></div></form>
            <?php
            $date = $_GET['date'] ?? $today_date;
            $path = sprintf('%s/attendance/%s/%s/%s/attendance-%s.ndjson', $ROOT_DB, substr($date,0,4), substr($date,5,2), substr($date,8,2), $date);
            echo '<div style="margin-top:15px; display:flex; justify-content:space-between; align-items:center;">';
            echo '<div class="small">File: <span class="code" style="padding:4px 8px; font-size:12px;">' . h(basename($path)) . '</span></div>';
            echo '<div class="small">Export: <a class="btn ghost" href="?export=attendance&date=' . h($date) . '&format=ndjson">NDJSON</a> <a class="btn ghost" href="?export=attendance&date=' . h($date) . '&format=csv">CSV</a></div>';
            echo '</div>';

            if (!file_exists($path)) {
                echo "<div class='card small' style='text-align:center; margin-top:15px; background:rgba(239,68,68,0.1)'>No attendance file found for " . h($date) . "</div>";
            } else {
                $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines === false) $lines = [];
                echo '<table class="table"><thead><tr><th>Time (12h)</th><th>Name</th><th>Admno</th><th>UID</th><th>Device</th><th>Result</th><th>Message</th></tr></thead><tbody>';
                foreach ($lines as $l) {
                    $o = json_decode($l, true);
                    if (!$o) continue;
                    $badge = ($o['result']??'')=='accepted' ? 'success' : 'error';
                    echo '<tr>';
                    echo '<td class="small">' . h(format_timestamp_12h($o['timestamp'] ?? '')) . '</td>';
                    echo '<td><strong>' . h($o['name'] ?? $o['user_id'] ?? '') . '</strong></td>';
                    echo '<td class="small">' . h($o['admno_stored'] ?? $o['admno_supplied'] ?? '') . '</td>';
                    echo '<td class="small">' . h($o['uid'] ?? '') . '</td>';
                    echo '<td class="small">' . h($o['device_id'] ?? '') . '</td>';
                    echo '<td><span class="badge ' . $badge . '">' . h(ucfirst($o['result'] ?? '')) . '</span></td>';
                    echo '<td class="small">' . h($o['message'] ?? '') . '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';
            }
            ?>
          </div>
        </div>

      <?php elseif ($page === 'indexes'): ?>
        <div class="card animate-fade">
          <div style="display:flex;justify-content:space-between;align-items:center"><div style="font-weight:700; font-size: 18px">Database Indexes (Cross-Reference Maps)</div><div><a class="btn" href="?op=regen_indexes" onclick="return confirmAction('WARNING: Are you sure you want to REGENERATE ALL INDEXES? This will scan all user files and overwrite existing index files.')">Regenerate All Indexes</a></div></div>
          <div class="small" style="margin-top:5px; margin-bottom:15px">These files map short identifiers (Admno, UID) to the unique system `user_id`. Regenerate if data integrity issues are suspected.</div>
          <div style="display:grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px">
              <div class="card" style="margin:0; background:rgba(0,0,0,0.2)">
                <div style="font-weight:600">UIDs Index (<?= count($indexes['uids']) ?> entries)</div>
                <div class="small">Maps Card UID to User ID.</div>
                <div class="code" style="margin-top:6px;max-height:200px;overflow:auto"><?= h(json_encode(array_slice($indexes['uids'], 0, 10), JSON_PRETTY_PRINT)) . (count($indexes['uids']) > 10 ? "\n..." : '') ?></div>
              </div>
              <div class="card" style="margin:0; background:rgba(0,0,0,0.2)">
                <div style="font-weight:600">Admission No. Index (<?= count($indexes['admnos']) ?> entries)</div>
                <div class="small">Maps Admission Number to User ID.</div>
                <div class="code" style="margin-top:6px;max-height:200px;overflow:auto"><?= h(json_encode(array_slice($indexes['admnos'], 0, 10), JSON_PRETTY_PRINT)) . (count($indexes['admnos']) > 10 ? "\n..." : '') ?></div>
              </div>
          </div>
        </div>

      <?php elseif ($page === 'import_export'): ?>
        <div class="card animate-fade">
          <div style="font-weight:700; font-size: 18px">Import User Records</div>
          <div class="small">Supported formats: JSON array or NDJSON (lines). Creates new user records. **Requires POST handler for file upload.**</div>
          <form method="post" enctype="multipart/form-data" style="margin-top:10px">
            <input type="hidden" name="op" value="import_users">
            <div class="form-row"><input type="file" name="import_file" class="input" required style="width:auto"></div>
            <div style="text-align:right;margin-top:8px"><button class="btn" disabled>Upload & Import Users (Not Implemented)</button></div>
          </form>
        </div>
        <div class="card animate-fade">
          <div style="font-weight:700; font-size: 18px">Export Data</div>
          <div class="small">Download a complete copy of user data or attendance logs.</div>
          <div style="margin-top:10px; display:flex; gap:10px; flex-wrap:wrap;">
            <a class="btn" href="?export=users&format=json">Users JSON (Full)</a>
            <a class="btn ghost" href="?export=users&format=csv">Users CSV (Table)</a>
            <a class="btn ghost" href="?export=attendance&date=<?= $today_date ?>&format=ndjson">Today Attendance NDJSON</a>
          </div>
        </div>

      <?php elseif ($page === 'backup'): ?>
        <div class="card animate-fade">
          <div style="font-weight:700; font-size: 18px">Database Backup</div>
          <div class="small">Create a complete ZIP archive of the entire `/database` directory. **Requires ZipArchive extension on server.**</div>
          <form method="post"><input type="hidden" name="op" value="backup_zip"><div style="text-align:right;margin-top:8px"><button class="btn" disabled>Create & Download Database ZIP (Not Implemented)</button></div></form>
        </div>
        <div class="card animate-fade">
          <div style="font-weight:700; font-size: 18px">Database Restore</div>
          <div class="alert alert-error">
              **DANGER: Manual Action Required.** Restoring a database from a ZIP file will completely **OVERWRITE** the current `/database` directory. This is not automated in this UI for safety reasons.
          </div>
        </div>

      <?php else: ?>
        <div class="card animate-fade">
            <div style="font-weight:700; font-size: 18px">Page Not Found</div>
            <div class="small">The requested page `<?= h($page) ?>` does not exist. Please use the navigation links.</div>
        </div>
      <?php endif; ?>

      <div class="footer">
          Titanium Pro Attendance System — Professional Admin UI • File-based DB • <?= date('Y-m-d H:i:s') ?>
      </div>
    </div>
  </main>
</div>
</body>
</html>