<?php
// attendance.php
// Logs attendance to /database/attendance/YYYY/MM/DD/*.ndjson and /database/attendance_by_user/user-<id>.ndjson
// Rejects duplicate accepted attendance for same user/day (returns status "already").
// Accepts GET or POST.

// --- unchanged logic except timestamps are now in Asia/Kolkata (IST) ---
header("Content-Type: application/json; charset=utf-8");

$BASE = __DIR__ . "/database";
$USERS = $BASE . "/users";
$ATT = $BASE . "/attendance";
$ATT_BY_USER = $BASE . "/attendance_by_user";
$IDX = $BASE . "/index";
foreach ([$USERS,$ATT,$ATT_BY_USER,$IDX] as $d) if (!is_dir($d)) mkdir($d, 0777, true);

function read_input(){ $in=$_POST; if(empty($in)){ $raw=@file_get_contents("php://input"); $json=@json_decode($raw,true); if(is_array($json)) $in=$json;} if(empty($in)) $in=$_GET; return $in?:[]; }
function normal_uid($s){ return strtoupper(preg_replace('/\s+/', '', (string)$s)); }
function append_line($path, $obj){ $line = json_encode($obj, JSON_UNESCAPED_SLASHES) . PHP_EOL; file_put_contents($path, $line, FILE_APPEND | LOCK_EX); }

// --- Set timezone to India Standard Time (IST)
date_default_timezone_set('Asia/Kolkata');

$in = read_input();
$user_id = isset($in['user_id']) ? trim((string)$in['user_id']) : null;
$uid_raw = isset($in['uid']) ? trim((string)$in['uid']) : null;
$admno   = isset($in['admno']) ? trim((string)$in['admno']) : null;
$device  = isset($in['device_id']) ? trim((string)$in['device_id']) : "";
$action  = isset($in['action']) ? trim((string)$in['action']) : "present";

if (!$user_id || !$uid_raw || !$admno) {
    http_response_code(400); echo json_encode(["status"=>"error","message"=>"user_id, uid, admno required"]); exit;
}

$uid = normal_uid($uid_raw);
$user_path = "$USERS/user-{$user_id}.json";
if (!file_exists($user_path)) { http_response_code(404); echo json_encode(["status"=>"error","message"=>"user not found"]); exit; }
$user = @json_decode(@file_get_contents($user_path), true); if(!is_array($user)){ http_response_code(500); echo json_encode(["status"=>"error","message"=>"invalid user file"]); exit; }

// Use IST for timestamp (ISO 8601 format with +05:30)
$now_dt = new DateTime('now', new DateTimeZone('Asia/Kolkata'));
$now = $now_dt->format(DateTime::ATOM);

$event = [
    "event_id"=>"evt_".bin2hex(random_bytes(5)),
    "timestamp"=>$now,
    "user_id"=>$user_id,
    "admno_supplied"=>$admno,
    "admno_stored"=>$user['admno'] ?? "",
    "name"=>$user['name'] ?? "",
    "class"=>$user['class'] ?? "",
    "uid"=>$uid,
    "device_id"=>$device,
    "action"=>$action,
    "result"=>null,
    "message"=>""
];

// day file (use IST date parts)
$dt = $now_dt;
$Y = $dt->format("Y"); $M = $dt->format("m"); $D = $dt->format("d");
$daydir = "$ATT/$Y/$M/$D"; if(!is_dir($daydir)) mkdir($daydir,0777,true);
$dayfile = "$daydir/attendance-$Y-$M-$D.ndjson";

// check existing accepted event for same user today
$existingAccepted = null;
if (file_exists($dayfile)) {
    $fh = fopen($dayfile, "r");
    if ($fh) {
        while (($line = fgets($fh)) !== false) {
            $o = @json_decode(trim($line), true);
            if (!is_array($o)) continue;
            if (isset($o['user_id']) && $o['user_id'] === $user_id && isset($o['result']) && $o['result'] === 'accepted') {
                $existingAccepted = $o; break;
            }
        }
        fclose($fh);
    }
}
if ($existingAccepted !== null) {
    echo json_encode([
        "status"=>"already",
        "message"=>"user already attended today",
        "event"=>$existingAccepted
    ], JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES);
    exit;
}

// validate admno
if ((string)$user['admno'] !== (string)$admno) {
    $event['result'] = 'rejected'; $event['message'] = 'admno mismatch';
} else {
    // check uid is assigned
    $assigned = false;
    foreach ($user['uids'] ?? [] as $u) { if (normal_uid($u) === $uid) { $assigned = true; break; } }
    if ($assigned) { $event['result'] = 'accepted'; $event['message'] = 'attendance recorded'; }
    else { $event['result'] = 'rejected'; $event['message'] = 'uid not assigned to this user'; }
}

// append to day file and per-user file
append_line($dayfile, $event);
$userdir = $ATT_BY_USER; if(!is_dir($userdir)) mkdir($userdir,0777,true);
$userfile = "$userdir/user-{$user_id}.ndjson";
append_line($userfile, $event);

// respond
echo json_encode(["status"=>"ok","event"=>$event], JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES);
