<?php
// create_user.php
// Create user: server generates user_id, saves user file and updates indexes.
// Accepts GET or POST (JSON or form).

header("Content-Type: application/json; charset=utf-8");

$BASE = __DIR__ . "/database";
$USERS = $BASE . "/users";
$IDX   = $BASE . "/index";
foreach ([$USERS, $IDX] as $d) if (!is_dir($d)) mkdir($d, 0777, true);

// Helpers
function read_input(){
    $in = $_POST;
    if (empty($in)) {
        $raw = @file_get_contents("php://input");
        $json = @json_decode($raw, true);
        if (is_array($json)) $in = $json;
    }
    if (empty($in)) $in = $_GET;
    return $in ?: [];
}
function normal_uid($s){ return strtoupper(preg_replace('/\s+/', '', (string)$s)); }
function atomic_save($path, $data){
    $tmp = $path . ".tmp";
    if (file_put_contents($tmp, json_encode($data, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES)) === false) return false;
    rename($tmp, $path);
    return true;
}
function load_index($p){ if(!file_exists($p)) return []; $c=@file_get_contents($p); $j=@json_decode($c,true); return is_array($j)?$j:[]; }

$in = read_input();
$admno = isset($in['admno']) ? trim((string)$in['admno']) : null;
$name  = isset($in['name'])  ? trim((string)$in['name']) : null;
$class = isset($in['class']) ? trim((string)$in['class']) : null;
$uid_raw = isset($in['uid']) ? trim((string)$in['uid']) : null;

if (!$admno || !$name || !$class) {
    http_response_code(400);
    echo json_encode(["status"=>"error","message"=>"admno, name, class are required"]);
    exit;
}
$uid = ($uid_raw !== null && $uid_raw !== "") ? normal_uid($uid_raw) : null;

// Load indexes
$names_p = "$IDX/names.json";
$admnos_p = "$IDX/admnos.json";
$uids_p = "$IDX/uids.json";
$names = load_index($names_p);
$admnos = load_index($admnos_p);
$uids = load_index($uids_p);

// Conflict checks
if (isset($admnos[$admno])) {
    http_response_code(409);
    echo json_encode(["status"=>"error","message"=>"admno already used"]);
    exit;
}
if ($uid !== null && isset($uids[$uid])) {
    http_response_code(409);
    echo json_encode(["status"=>"error","message"=>"uid already assigned"]);
    exit;
}

// Generate user_id
$user_id = "usr_" . bin2hex(random_bytes(4));
$user_path = "$USERS/user-{$user_id}.json";

$now = date("c");
$user = [
    "user_id" => $user_id,
    "admno"   => $admno,
    "name"    => $name,
    "class"   => $class,
    "uids"    => $uid ? [$uid] : [],
    "created_at" => $now,
    "updated_at" => $now
];

// Save user file
if (!atomic_save($user_path, $user)) {
    http_response_code(500);
    echo json_encode(["status"=>"error","message"=>"failed to write user file"]);
    exit;
}

// Update indexes (atomic)
$names[$user_id] = $name;
$admnos[$admno] = $user_id;
if ($uid !== null) $uids[$uid] = $user_id;
atomic_save($names_p, $names);
atomic_save($admnos_p, $admnos);
atomic_save($uids_p, $uids);

// Response (separated clarity)
http_response_code(201);
echo json_encode([
    "status" => "ok",
    "message" => "user created",
    "user_details" => [
        "user_id" => $user_id,
        "name" => $name,
        "admno" => $admno,
        "class" => $class,
        "card_id" => $uid ?? null,
        "card_ids" => $uid ? [$uid] : []
    ],
    "data" => $user
], JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES);
