<?php
header('Content-Type: application/json');

// Allow browser to call your script from anywhere
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Get the batch_id from the URL path
$path_info = isset($_SERVER['PATH_INFO']) ? trim($_SERVER['PATH_INFO'], '/') : null;

if (!$path_info) {
    echo json_encode([
        "success" => false,
        "error"   => "No batch_id provided. Use like: script.php/<batch_id>"
    ], JSON_PRETTY_PRINT);
    exit;
}

$batch_id = $path_info;

// Build the CORS proxy URL (protects real API)
$cors_base = "https://cors.hideme.eu.org/?url=";
$real_url = "https://api.pwthor.site/api/batch/" . urlencode($batch_id);
$url = $cors_base . urlencode($real_url);

// Setup headers for the API request
$headers = [
    // DON'T advertise encodings your libcurl doesn't support (remove br, zstd)
    "accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.7",
    // remove custom Accept-Encoding header so cURL can set supported encodings via CURLOPT_ENCODING
    //"accept-encoding: gzip, deflate, br, zstd",
    "accept-language: en-GB,en-US;q=0.9,en;q=0.8,hi;q=0.7",
    "cache-control: max-age=0",
    // It's better not to include an outdated cookie unless required
    // "cookie: ...",
    "priority: u=0, i",
    "sec-ch-ua: \"Not;A=Brand\";v=\"99\", \"Google Chrome\";v=\"139\", \"Chromium\";v=\"139\"",
    "sec-ch-ua-platform: \"Windows\"",
    "user-agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/139.0.0.0 Safari/537.36"
];

// Initialize cURL
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

// Let libcurl handle encodings it actually supports.
// Use 'gzip, deflate' (common), NOT br/zstd unless your libcurl supports brotli/zstd.
curl_setopt($ch, CURLOPT_ENCODING, "gzip, deflate");

// follow redirects if the proxy issues them
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);

// reasonable timeouts
curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);

$response = curl_exec($ch);

if (curl_errno($ch)) {
    echo json_encode([
        "success" => false,
        "error"   => "cURL error: " . curl_error($ch)
    ], JSON_PRETTY_PRINT);
    curl_close($ch);
    exit;
}

$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

// Treat 200 and 304 as OK responses (304 = Not Modified)
if ($http_code === 200) {
    // Try to parse JSON
    $data = json_decode($response, true);
    if (json_last_error() === JSON_ERROR_NONE) {
        echo json_encode([
            "success" => true,
            "data"    => $data
        ], JSON_PRETTY_PRINT);
    } else {
        // If the proxy returned HTML (or something else), return raw
        echo json_encode([
            "success" => false,
            "error"   => "Invalid JSON response from API",
            "raw"     => $response
        ], JSON_PRETTY_PRINT);
    }
    exit;
} elseif ($http_code === 304) {
    // Not Modified - proxy returned 304 (Etag matched). This is not an error.
    echo json_encode([
        "success"     => true,
        "not_modified"=> true,
        "http_code"   => 304,
        "message"     => "Resource not modified (304). Use cached copy if available."
    ], JSON_PRETTY_PRINT);
    exit;
} else {
    echo json_encode([
        "success" => false,
        "error"   => "API request failed with HTTP code $http_code",
        "raw"     => $response
    ], JSON_PRETTY_PRINT);
    exit;
}
?>
