<?php
// Allow access from any origin (CORS)
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Get the path after the script name (PATH_INFO) or fallback to query params
$path = isset($_SERVER['PATH_INFO']) ? trim($_SERVER['PATH_INFO'], '/') : '';
$parts = ($path === '') ? [] : explode('/', $path);

if (count($parts) === 2) {
    $batch_id = $parts[0];
    $schedule_id = $parts[1];
} else {
    // fallback: accept ?batch_id=...&schedule_id=...
    $batch_id = isset($_GET['batch_id']) ? $_GET['batch_id'] : null;
    $schedule_id = isset($_GET['schedule_id']) ? $_GET['schedule_id'] : null;
}

if (!$batch_id || !$schedule_id) {
    echo json_encode([
        "success" => false,
        "error" => "batch_id and schedule_id are required (use /script.php/<batch_id>/<schedule_id> or ?batch_id=...&schedule_id=...)"
    ], JSON_PRETTY_PRINT);
    exit();
}

// Build the inner API URL (the actual API you want to call)
$inner_api = "https://api.pwthor.site/api/url"
           . "?batch_id=" . rawurlencode($batch_id)
           . "&schedule_id=" . rawurlencode($schedule_id);

// Proxy base
$proxy_base = "https://cors.hideme.eu.org/?url=";

// IMPORTANT: encode the entire inner URL before appending to the proxy
$request_url = $proxy_base . rawurlencode($inner_api);

/**
 * Fetch URL with cURL and return structured info
 */
function fetch_data($url) {
    $ch = curl_init();

    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    // include response headers with body so we can inspect ETag / Content-Encoding
    curl_setopt($ch, CURLOPT_HEADER, true);

    // timeouts & redirects
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_TIMEOUT, 25);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);

    // Limit encodings to avoid brotli/zstd issues on older libcurl builds
    curl_setopt($ch, CURLOPT_ENCODING, "gzip, deflate");

    // verify peer in production; set to false only for temporary debugging
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);

    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Accept: application/json",
        "User-Agent: Mozilla/5.0 (PHP cURL)"
    ]);

    $raw = curl_exec($ch);
    $errno = curl_errno($ch);
    $errstr = curl_error($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $header_size = curl_getinfo($ch, CURLINFO_HEADER_SIZE);

    // Split headers/body
    $headers_raw = '';
    $body = null;
    if ($raw !== false && $header_size > 0) {
        $headers_raw = substr($raw, 0, $header_size);
        $body = substr($raw, $header_size);
    } else {
        $body = $raw;
    }

    curl_close($ch);

    return [
        'errno' => $errno,
        'errstr' => $errstr,
        'http_code' => $http_code,
        'headers_raw' => $headers_raw,
        'body' => $body
    ];
}

// Perform request
$result = fetch_data($request_url);

// Check for cURL error
if ($result['errno'] !== 0) {
    echo json_encode([
        "success" => false,
        "error" => "cURL error: ({$result['errno']}) {$result['errstr']}"
    ], JSON_PRETTY_PRINT);
    exit();
}

// If proxy returned 304 Not Modified, return a non-error so client can use cache
if ($result['http_code'] === 304) {
    echo json_encode([
        "success" => true,
        "http_code" => 304,
        "not_modified" => true,
        "headers" => $result['headers_raw']
    ], JSON_PRETTY_PRINT);
    exit();
}

// If API returned non-200, return the raw body and headers to help debugging
if ($result['http_code'] !== 200) {
    // Try to decode JSON error body if any
    $body_json = json_decode($result['body'], true);
    echo json_encode([
        "success" => false,
        "error" => "API request failed with HTTP code " . $result['http_code'],
        "http_code" => $result['http_code'],
        "headers" => $result['headers_raw'],
        "raw" => $result['body'],
        "parsed_body" => $body_json === null ? null : $body_json
    ], JSON_PRETTY_PRINT);
    exit();
}

// Parse success response
$data = json_decode($result['body'], true);
if (json_last_error() !== JSON_ERROR_NONE) {
    echo json_encode([
        "success" => false,
        "error" => "Invalid JSON response from API",
        "json_error" => json_last_error_msg(),
        "headers" => $result['headers_raw'],
        "raw" => $result['body']
    ], JSON_PRETTY_PRINT);
    exit();
}

// Validate required fields
if (!isset($data['signed_url']) || !isset($data['video_id'])) {
    echo json_encode([
        "success" => false,
        "error" => "API response missing 'signed_url' or 'video_id'.",
        "raw_response" => $data,
        "headers" => $result['headers_raw']
    ], JSON_PRETTY_PRINT);
    exit();
}

// Build player URL
$player_url = "https://pwthor.site/pwplayer.html?encrypted=" . rawurlencode($data['signed_url']) . "&iv=" . rawurlencode($data['video_id']);

echo json_encode([
    "success" => true,
    "player_url" => $player_url,
    "headers" => $result['headers_raw']
], JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
