<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header("Content-Type: application/json");

// Utility to send an error JSON + exit
function send_json_error($message, $code = 400, $extra = []) {
    http_response_code($code);
    $payload = array_merge([
        "success" => false,
        "error"   => $message
    ], $extra);
    echo json_encode($payload, JSON_PRETTY_PRINT);
    exit;
}

// Handle preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

// Get params (either query params or POST)
$batch_id     = $_GET['batch_id']     ?? $_POST['batch_id']     ?? null;
$subject_slug = $_GET['subject_slug'] ?? $_POST['subject_slug'] ?? null;
$schedule_id  = $_GET['schedule_id']  ?? $_POST['schedule_id']  ?? null;

if (!$batch_id || !$subject_slug || !$schedule_id) {
    send_json_error("Missing required parameters: batch_id, subject_slug, schedule_id", 422);
}

// Build inner API URL and encode it for the proxy
$inner = "https://api.pwthor.site/api/batch/"
       . rawurlencode($batch_id)
       . "/subject/"
       . rawurlencode($subject_slug)
       . "/schedule/"
       . rawurlencode($schedule_id)
       . "/content";

$proxy_base = "https://cors.hideme.eu.org/?url=";
$request_url = $proxy_base . rawurlencode($inner);

// cURL: call the proxy which calls the real API
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $request_url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

// Include response headers so we can inspect status/encoding if needed
curl_setopt($ch, CURLOPT_HEADER, true);

// follow redirects if proxy redirects
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);

// Timeouts and SSL
curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 8);
curl_setopt($ch, CURLOPT_TIMEOUT, 25);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);

// IMPORTANT: request only encodings most libcurl builds support to avoid errors.
// If you know your environment supports Brotli, you can use "" (empty) instead.
curl_setopt($ch, CURLOPT_ENCODING, "gzip, deflate");

// Request headers (do not advertise br/zstd)
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    "Accept: application/json",
    "User-Agent: Mozilla/5.0 (PHP cURL)"
]);

$raw = curl_exec($ch);
$curl_errno = curl_errno($ch);
$curl_error = curl_error($ch);
$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$header_size = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
curl_close($ch);

// cURL error
if ($curl_errno !== 0) {
    send_json_error("cURL error: ({$curl_errno}) {$curl_error}", 500, [
        "request_url" => $request_url
    ]);
}

// Split headers/body
$headers_raw = '';
$body = $raw;
if ($raw !== false && $header_size > 0) {
    $headers_raw = substr($raw, 0, $header_size);
    $body = substr($raw, $header_size);
}

// Handle 304 Not Modified gracefully
if ($http_code === 304) {
    echo json_encode([
        "success" => true,
        "http_code" => 304,
        "not_modified" => true,
        "message" => "Resource not modified. Use cached copy if available.",
        "headers" => $headers_raw
    ], JSON_PRETTY_PRINT);
    exit;
}

// Non-200 from proxy/API
if ($http_code !== 200) {
    // Try to parse body as JSON for better debugging
    $parsed_body = json_decode($body, true);
    send_json_error("API returned HTTP code {$http_code}", $http_code, [
        "headers" => $headers_raw,
        "raw" => $body,
        "parsed_body" => ($parsed_body === null ? null : $parsed_body),
        "request_url" => $request_url
    ]);
}

// Decode JSON body
$data = json_decode($body, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    send_json_error("Invalid JSON response from API: " . json_last_error_msg(), 502, [
        "headers" => $headers_raw,
        "raw" => $body
    ]);
}

// Expect structure: { "data": { ... } }
if (!isset($data['data']) || !is_array($data['data'])) {
    send_json_error("API response missing expected 'data' object", 502, [
        "raw_response" => $data
    ]);
}

$item = $data['data'];

// Build filtered response safely (use null defaults)
$pdf_url = null;
if (isset($item['homeworkIds'][0]['attachmentIds'][0])) {
    $att = $item['homeworkIds'][0]['attachmentIds'][0];
    // Some APIs return baseUrl ending with / or not — join safely
    $base = rtrim($att['baseUrl'] ?? '', '/');
    $key  = ltrim($att['key'] ?? '', '/');
    if ($base !== '' && $key !== '') {
        $pdf_url = $base . '/' . $key;
    } elseif (!empty($att['baseUrl'])) {
        $pdf_url = $att['baseUrl'] . ($att['key'] ?? '');
    }
}

$result = [
    "success" => true,
    "topic"   => $item['topic'] ?? null,
    "note"    => $item['homeworkIds'][0]['note'] ?? null,
    "status"  => $item['status'] ?? null,
    "date"    => $item['date'] ?? null,
    "subject" => $item['subject']['name'] ?? null,
    "pdf_url" => $pdf_url
];

// Return clean response (no raw headers by default)
http_response_code(200);
echo json_encode($result, JSON_PRETTY_PRINT);
