<?php
// Allow CORS for all domains
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header('Content-Type: application/json');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

/*
  Accept batch_id and subject_id either via PATH_INFO:
    /script.php/<batch_id>/<subject_id>
  or via query parameters:
    script.php?batch_id=...&subject_id=...
*/

// Get from PATH_INFO if available
$path_info = isset($_SERVER['PATH_INFO']) ? trim($_SERVER['PATH_INFO'], '/') : null;

if ($path_info) {
    $parts = explode('/', $path_info);
    if (count($parts) >= 2) {
        $batch_id = $parts[0];
        $subject_id = $parts[1];
    }
}

// Fallback to query parameters
if (empty($batch_id) || empty($subject_id)) {
    $batch_id = isset($_GET['batch_id']) ? $_GET['batch_id'] : (isset($_POST['batch_id']) ? $_POST['batch_id'] : null);
    $subject_id = isset($_GET['subject_id']) ? $_GET['subject_id'] : (isset($_POST['subject_id']) ? $_POST['subject_id'] : null);
}

if (!$batch_id || !$subject_id) {
    echo json_encode([
        "success" => false,
        "error"   => "No batch_id or subject_id provided. Use: script.php/<batch_id>/<subject_id> or script.php?batch_id=...&subject_id=..."
    ], JSON_PRETTY_PRINT);
    exit;
}

/* ------------------ CHANGE ONLY BELOW ------------------ */

// Whether to use the public CORS proxy; set to false to call API directly
$use_proxy = true;
$proxy_base = "https://cors.hideme.eu.org/?url=";

// Base API URL
$api_path = "/api/batch/" . rawurlencode($batch_id) . "/subject/" . rawurlencode($subject_id) . "/topics";
$real_url = "https://api.pwthor.site" . $api_path;

if ($use_proxy) {
    // wrap the real URL in the proxy
    $request_url = $proxy_base . rawurlencode($real_url);
} else {
    $request_url = $real_url;
}

// Setup request headers
// Do NOT include an Accept-Encoding header advertising unsupported encodings.
// Let libcurl handle supported encodings via CURLOPT_ENCODING.
$headers = [
    "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/139.0.0.0 Safari/537.36",
    "Accept: application/json, text/plain, */*"
];

// Optionally provide an If-None-Match to intentionally trigger 304 behavior (comment out if not needed).
// If you want to avoid 304 completely, simply don't send If-None-Match or ETag-related headers.
// $etag = ''; // e.g. 'W/"abc123"'
// if ($etag) { $headers[] = "If-None-Match: " . $etag; }

/* -------------------------------------------------------- */

// Initialize cURL
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $request_url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
// separate headers from body if you want to inspect them
curl_setopt($ch, CURLOPT_HEADER, false);

// Let libcurl request/handle common encodings it supports.
// Use "gzip, deflate" which is widely supported. If your cURL supports brotli/zstd and you want to use them, set "" (empty) instead.
curl_setopt($ch, CURLOPT_ENCODING, "gzip, deflate");

// follow redirects if proxy or API redirects
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);

// timeouts for robustness
curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);

// set to true in production; left false if self-signed or debugging
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);

// Execute request
$response = curl_exec($ch);

if (curl_errno($ch)) {
    $err = curl_error($ch);
    $errno = curl_errno($ch);
    curl_close($ch);
    echo json_encode([
        "success" => false,
        "error"   => "cURL error: ($errno) $err"
    ], JSON_PRETTY_PRINT);
    exit;
}

$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

// Treat 200 OK and 304 Not Modified as non-errors
if ($http_code === 200) {
    // Attempt to decode JSON response
    $data = json_decode($response, true);
    if (json_last_error() === JSON_ERROR_NONE) {
        echo json_encode([
            "success" => true,
            "http_code" => 200,
            "data"    => $data
        ], JSON_PRETTY_PRINT);
        exit;
    } else {
        // API returned something that's not valid JSON
        echo json_encode([
            "success" => false,
            "http_code" => 200,
            "error"   => "Invalid JSON response from API",
            "raw"     => $response
        ], JSON_PRETTY_PRINT);
        exit;
    }
} elseif ($http_code === 304) {
    // Not Modified — proxy/API indicates cached copy may be used
    echo json_encode([
        "success"      => true,
        "http_code"    => 304,
        "not_modified" => true,
        "message"      => "Resource not modified (304). Use cached copy if available."
    ], JSON_PRETTY_PRINT);
    exit;
} else {
    // Other non-OK responses — return raw to help debugging
    echo json_encode([
        "success" => false,
        "http_code" => $http_code,
        "error"   => "API request failed with HTTP code $http_code",
        "raw"     => $response
    ], JSON_PRETTY_PRINT);
    exit;
}
?>
