<?php
/**
 * ================================================================
 * LIVE CLASS NOTIFICATION SYSTEM - COMPLETE CRON JOB (FIXED)
 * ================================================================
 * 
 * Features:
 * - Auto-sync users from users.json
 * - Fetch schedules from API
 * - Send notifications (scheduled, 1hr, 10min, started)
 * - Handle reschedules and cancellations
 * - Duplicate prevention
 * - User preference management
 * - Complete logging system
 * 
 * @version 2.0.1 (Fixed)
 * @author Notification System
 */

// ================================================================
// CONFIGURATION & INITIALIZATION
// ================================================================

error_reporting(E_ALL);
ini_set('display_errors', 1);
set_time_limit(300);

define('BASE_DIR', __DIR__);
define('DATA_DIR', BASE_DIR . '/data');
define('USERS_FILE', BASE_DIR . '/users.json');
define('LOG_DIR', BASE_DIR . '/logs');

// ================================================================
// MAIN EXECUTION
// ================================================================

$startTime = microtime(true);
$logger = new Logger();
$logger->info("=== CRON JOB STARTED ===");

try {
    SystemInitializer::initialize();
    $logger->info("✓ System initialized");
    
    $settings = FileHandler::loadSettings();
    
    if (!$settings['global_settings']['system_enabled']) {
        $logger->warning("⚠ System disabled. Exiting.");
        exit;
    }
    
    if ($settings['global_settings']['maintenance_mode']) {
        $logger->warning("⚠ Maintenance mode. Exiting.");
        exit;
    }
    
    $userManager = new UserManager();
    $syncResults = $userManager->syncUsers();
    $logger->info("✓ User sync: +{$syncResults['added']} ~{$syncResults['updated']} -{$syncResults['deleted']}");
    
    $activeUsers = $userManager->getActiveUsers();
    $logger->info("✓ Active users: " . count($activeUsers));
    
    $batchIds = $userManager->getAllBatchIds($activeUsers);
    $logger->info("✓ Batches to check: " . count($batchIds));
    
    $notificationEngine = new NotificationEngine($settings, $logger);
    $stats = ['sent' => 0, 'edited' => 0, 'cancelled' => 0];
    
    foreach ($batchIds as $batchId) {
        $logger->info("→ Processing batch: $batchId");
        
        $schedules = APIClient::fetchBatchSchedule($batchId, $settings);
        
        if (empty($schedules)) {
            $logger->info("  No schedules found");
            continue;
        }
        
        $logger->info("  Found " . count($schedules) . " schedules");
        
        foreach ($schedules as $schedule) {
            $result = $notificationEngine->processSchedule($schedule, $activeUsers, $userManager);
            $stats['sent'] += $result['sent'];
            $stats['edited'] += $result['edited'];
            $stats['cancelled'] += $result['cancelled'];
        }
    }
    
    FileHandler::cleanup($settings);
    
    $executionTime = round(microtime(true) - $startTime, 2);
    $logger->info("=== COMPLETED ===", [
        'time' => $executionTime . 's',
        'sent' => $stats['sent'],
        'edited' => $stats['edited'],
        'cancelled' => $stats['cancelled']
    ]);
    
} catch (Exception $e) {
    $logger->error("CRITICAL ERROR: " . $e->getMessage());
    $logger->error("File: " . $e->getFile() . " Line: " . $e->getLine());
}

$logger->close();

// ================================================================
// SYSTEM INITIALIZER CLASS
// ================================================================

class SystemInitializer {
    public static function initialize() {
        $dirs = [
            DATA_DIR,
            DATA_DIR . '/users',
            DATA_DIR . '/schedules',
            DATA_DIR . '/notifications',
            DATA_DIR . '/pending',
            DATA_DIR . '/trash',
            LOG_DIR
        ];
        
        foreach ($dirs as $dir) {
            if (!file_exists($dir)) {
                mkdir($dir, 0755, true);
            }
        }
        
        if (!file_exists(DATA_DIR . '/settings.json')) {
            self::createDefaultSettings();
        }
        
        $defaultFiles = [
            DATA_DIR . '/notifications/notification_log.json' => [],
            DATA_DIR . '/pending/pending_notifications.json' => [],
            DATA_DIR . '/schedules/schedule_cache.json' => [],
            DATA_DIR . '/trash/trash.json' => ['deleted_users' => [], 'deleted_schedules' => []]
        ];
        
        foreach ($defaultFiles as $file => $defaultData) {
            if (!file_exists($file)) {
                FileHandler::saveJSON($file, $defaultData);
            }
        }
    }
    
    private static function createDefaultSettings() {
        $settings = [
            'global_settings' => [
                'system_enabled' => true,
                'default_notifications_enabled' => true,
                'maintenance_mode' => false,
                'timezone' => 'Asia/Kolkata',
                'auto_sync_users' => true
            ],
            'notification_timings' => [
                'on_schedule' => ['enabled' => true, 'offset_minutes' => 0, 'label' => '📅 Class Scheduled'],
                'before_1_hour' => ['enabled' => true, 'offset_minutes' => -60, 'label' => '⏰ Starting in 1 Hour'],
                'before_10_min' => ['enabled' => true, 'offset_minutes' => -10, 'label' => '🔔 Starting in 10 Minutes'],
                'on_start' => ['enabled' => true, 'offset_minutes' => 0, 'label' => '▶️ Class Started']
            ],
            'notification_types' => [
                'scheduled' => true,
                'rescheduled' => true,
                'cancelled' => true,
                'started' => true
            ],
            'message_handling' => [
                'edit_on_reschedule' => true,
                'delete_on_cancel' => false,
                'show_reschedule_history' => true
            ],
            'rate_limiting' => [
                'messages_per_second' => 25,
                'batch_delay_ms' => 50,
                'retry_failed' => true,
                'max_retries' => 3
            ],
            'cleanup' => [
                'log_retention_days' => 30,
                'notification_log_retention_days' => 90,
                'trash_retention_days' => 60,
                'auto_cleanup' => true
            ],
            'api' => [
                'schedule_api_url' => 'https://api.titaniumtech.site/educational/pw/todays_schedule_batch.php/',
                'request_timeout' => 30
            ],
            'telegram' => [
                'bot_token' => '8578809703:AAHL2xW_WOZ6_xdptqjm2dSFG0EQo1Q254U',
                'api_url' => 'https://api.telegram.org/bot'
            ]
        ];
        
        FileHandler::saveJSON(DATA_DIR . '/settings.json', $settings);
    }
}

// ================================================================
// FILE HANDLER CLASS
// ================================================================

class FileHandler {
    public static function loadJSON($filepath, $default = null) {
        if (!file_exists($filepath)) return $default;
        
        $content = file_get_contents($filepath);
        if ($content === false) return $default;
        
        $data = json_decode($content, true);
        return $data ?? $default;
    }
    
    public static function saveJSON($filepath, $data) {
        $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        $tempFile = $filepath . '.tmp';
        
        if (file_put_contents($tempFile, $json, LOCK_EX) === false) {
            return false;
        }
        
        return rename($tempFile, $filepath);
    }
    
    public static function loadSettings() {
        return self::loadJSON(DATA_DIR . '/settings.json', []);
    }
    
    public static function cleanup($settings) {
        if (!$settings['cleanup']['auto_cleanup']) return;
        
        $logFiles = glob(LOG_DIR . '/cron_*.log');
        $cutoff = time() - ($settings['cleanup']['log_retention_days'] * 86400);
        
        foreach ($logFiles as $file) {
            if (filemtime($file) < $cutoff) {
                unlink($file);
            }
        }
        
        $notifLog = self::loadJSON(DATA_DIR . '/notifications/notification_log.json', []);
        $notifCutoff = time() - ($settings['cleanup']['notification_log_retention_days'] * 86400);
        
        $notifLog = array_filter($notifLog, function($entry) use ($notifCutoff) {
            return strtotime($entry['sent_at'] ?? 'now') > $notifCutoff;
        });
        
        self::saveJSON(DATA_DIR . '/notifications/notification_log.json', array_values($notifLog));
    }
}

// ================================================================
// USER MANAGER CLASS
// ================================================================

class UserManager {
    private $logger;
    
    public function __construct() {
        $this->logger = new Logger();
    }
    
    public function syncUsers() {
        $stats = ['added' => 0, 'updated' => 0, 'deleted' => 0, 'unchanged' => 0, 'errors' => 0];
        
        if (!file_exists(USERS_FILE)) {
            throw new Exception("users.json not found");
        }
        
        $usersData = FileHandler::loadJSON(USERS_FILE, []);
        $currentUsers = [];
        
        foreach ($usersData as $user) {
            if (isset($user['telegram_id'])) {
                $currentUsers[$user['telegram_id']] = $user;
            }
        }
        
        $existingUsers = $this->getAllUserFiles();
        
        foreach ($currentUsers as $telegramId => $userData) {
            try {
                if (!isset($existingUsers[$telegramId])) {
                    $this->createUserFile($telegramId, $userData);
                    $stats['added']++;
                } else {
                    if ($this->updateUserFile($telegramId, $userData, $existingUsers[$telegramId])) {
                        $stats['updated']++;
                    } else {
                        $stats['unchanged']++;
                    }
                }
            } catch (Exception $e) {
                $stats['errors']++;
                $this->logger->error("Error processing user $telegramId: " . $e->getMessage());
            }
        }
        
        foreach ($existingUsers as $telegramId => $existingData) {
            if (!isset($currentUsers[$telegramId])) {
                $this->deleteUserFile($telegramId);
                $stats['deleted']++;
            }
        }
        
        return $stats;
    }
    
    private function createUserFile($telegramId, $sourceData) {
        $settings = FileHandler::loadSettings();
        
        $userDetails = [
            'telegram_id' => $telegramId,
            'username' => $sourceData['username'] ?? '',
            'name' => $sourceData['name'] ?? '',
            'class' => $sourceData['class'] ?? '',
            'age' => $sourceData['age'] ?? '',
            'created_at' => $sourceData['created_at'] ?? date('Y-m-d\TH:i:sP'),
            'bot_started' => false,
            'bot_blocked' => $sourceData['blocked'] ?? false,
            'global_notifications_enabled' => $settings['global_settings']['default_notifications_enabled'],
            'notification_preferences' => [
                'scheduled' => true,
                'before_1_hour' => true,
                'before_10_min' => true,
                'on_start' => true,
                'rescheduled' => true,
                'cancelled' => true
            ],
            'batch_settings' => $this->createBatchSettings($sourceData),
            'preferences' => [
                'timezone' => $settings['global_settings']['timezone'],
                'quiet_hours' => ['enabled' => false, 'start' => '22:00', 'end' => '07:00']
            ],
            'last_synced' => date('Y-m-d H:i:s'),
            'total_notifications_received' => 0
        ];
        
        FileHandler::saveJSON(DATA_DIR . "/users/{$telegramId}.json", $userDetails);
    }
    
    private function createBatchSettings($sourceData) {
        $batchSettings = [];
        
        if (isset($sourceData['subscriptions']) && is_array($sourceData['subscriptions'])) {
            foreach ($sourceData['subscriptions'] as $sub) {
                if (isset($sub['batch']['_id'])) {
                    $batchId = $sub['batch']['_id'];
                    $batchSettings[$batchId] = [
                        'batch_id' => $batchId,
                        'batch_name' => $sub['batch']['name'] ?? '',
                        'notifications_enabled' => true,
                        'notification_types' => [
                            'scheduled' => true,
                            'before_1_hour' => true,
                            'before_10_min' => true,
                            'on_start' => true,
                            'rescheduled' => true,
                            'cancelled' => true
                        ],
                        'subscription_start' => $sub['start'] ?? null,
                        'subscription_end' => $sub['end'] ?? null
                    ];
                }
            }
        }
        
        return $batchSettings;
    }
    
    private function updateUserFile($telegramId, $sourceData, $existingData) {
        $updated = false;
        
        $fields = ['username', 'name', 'class', 'age'];
        foreach ($fields as $field) {
            if (isset($sourceData[$field]) && $sourceData[$field] !== ($existingData[$field] ?? '')) {
                $existingData[$field] = $sourceData[$field];
                $updated = true;
            }
        }
        
        if (isset($sourceData['blocked']) && $sourceData['blocked'] !== ($existingData['bot_blocked'] ?? false)) {
            $existingData['bot_blocked'] = $sourceData['blocked'];
            $updated = true;
        }
        
        $newBatchSettings = $this->createBatchSettings($sourceData);
        
        foreach ($newBatchSettings as $batchId => $batchData) {
            if (!isset($existingData['batch_settings'][$batchId])) {
                $existingData['batch_settings'][$batchId] = $batchData;
                $updated = true;
            } else {
                $existing = $existingData['batch_settings'][$batchId];
                if ($batchData['batch_name'] !== $existing['batch_name'] ||
                    $batchData['subscription_start'] !== $existing['subscription_start'] ||
                    $batchData['subscription_end'] !== $existing['subscription_end']) {
                    
                    $existingData['batch_settings'][$batchId]['batch_name'] = $batchData['batch_name'];
                    $existingData['batch_settings'][$batchId]['subscription_start'] = $batchData['subscription_start'];
                    $existingData['batch_settings'][$batchId]['subscription_end'] = $batchData['subscription_end'];
                    $updated = true;
                }
            }
        }
        
        foreach ($existingData['batch_settings'] as $batchId => $batchData) {
            if (!isset($newBatchSettings[$batchId])) {
                $existingData['batch_settings'][$batchId]['notifications_enabled'] = false;
                $existingData['batch_settings'][$batchId]['removed_at'] = date('Y-m-d H:i:s');
                $updated = true;
            }
        }
        
        if ($updated) {
            $existingData['last_synced'] = date('Y-m-d H:i:s');
            FileHandler::saveJSON(DATA_DIR . "/users/{$telegramId}.json", $existingData);
        }
        
        return $updated;
    }
    
    private function deleteUserFile($telegramId) {
        $filepath = DATA_DIR . "/users/{$telegramId}.json";
        
        if (file_exists($filepath)) {
            $userData = FileHandler::loadJSON($filepath, []);
            
            $trash = FileHandler::loadJSON(DATA_DIR . '/trash/trash.json', ['deleted_users' => []]);
            $trash['deleted_users'][] = [
                'telegram_id' => $telegramId,
                'data' => $userData,
                'deleted_at' => date('Y-m-d H:i:s')
            ];
            FileHandler::saveJSON(DATA_DIR . '/trash/trash.json', $trash);
            
            unlink($filepath);
        }
    }
    
    private function getAllUserFiles() {
        $userDir = DATA_DIR . '/users';
        $files = glob($userDir . '/*.json');
        $users = [];
        
        foreach ($files as $file) {
            $telegramId = basename($file, '.json');
            $users[$telegramId] = FileHandler::loadJSON($file, []);
        }
        
        return $users;
    }
    
    public function getActiveUsers() {
        $allUsers = $this->getAllUserFiles();
        $activeUsers = [];
        
        foreach ($allUsers as $telegramId => $userData) {
            if (!empty($userData['bot_started']) &&
                empty($userData['bot_blocked']) &&
                !empty($userData['global_notifications_enabled'])) {
                $activeUsers[$telegramId] = $userData;
            }
        }
        
        return $activeUsers;
    }
    
    public function getAllBatchIds($activeUsers) {
        $batchIds = [];
        
        foreach ($activeUsers as $userData) {
            if (isset($userData['batch_settings'])) {
                foreach ($userData['batch_settings'] as $batchId => $batchData) {
                    if (!empty($batchData['notifications_enabled'])) {
                        $batchIds[$batchId] = true;
                    }
                }
            }
        }
        
        return array_keys($batchIds);
    }
    
    public function shouldReceiveNotification($userData, $batchId, $notificationType) {
        if (empty($userData['global_notifications_enabled'])) return false;
        if (!isset($userData['batch_settings'][$batchId])) return false;
        
        $batchSettings = $userData['batch_settings'][$batchId];
        
        if (empty($batchSettings['notifications_enabled'])) return false;
        
        if (isset($batchSettings['notification_types'][$notificationType]) &&
            empty($batchSettings['notification_types'][$notificationType])) {
            return false;
        }
        
        if (isset($userData['notification_preferences'][$notificationType]) &&
            empty($userData['notification_preferences'][$notificationType])) {
            return false;
        }
        
        return true;
    }
}

// ================================================================
// API CLIENT CLASS
// ================================================================

class APIClient {
    public static function fetchBatchSchedule($batchId, $settings) {
        $url = $settings['api']['schedule_api_url'] . $batchId;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, $settings['api']['request_timeout']);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200 || !$response) {
            return [];
        }
        
        $data = json_decode($response, true);
        
        if (isset($data['data']['data']) && is_array($data['data']['data'])) {
            return $data['data']['data'];
        } elseif (isset($data['data']) && is_array($data['data'])) {
            return $data['data'];
        }
        
        return [];
    }
}

// ================================================================
// NOTIFICATION ENGINE CLASS
// ================================================================

class NotificationEngine {
    private $settings;
    private $logger;
    
    public function __construct($settings, $logger) {
        $this->settings = $settings;
        $this->logger = $logger;
    }
    
    public function processSchedule($schedule, $activeUsers, $userManager) {
        $scheduleId = $schedule['_id'];
        
        $result = ['sent' => 0, 'edited' => 0, 'cancelled' => 0];
        
        $status = $this->detectScheduleChanges($schedule);
        
        switch ($status['type']) {
            case 'new':
                $this->logger->info("  → New: $scheduleId");
                $this->createNotificationSet($schedule);
                break;
                
            case 'rescheduled':
                $this->logger->info("  → Rescheduled: $scheduleId");
                $result['edited'] += $this->handleReschedule($schedule, $activeUsers, $userManager, $status['old']);
                break;
                
            case 'cancelled':
                $this->logger->info("  → Cancelled: $scheduleId");
                $result['cancelled'] += $this->handleCancellation($schedule);
                break;
                
            case 'unchanged':
                $result['sent'] += $this->processPendingNotifications($schedule, $activeUsers, $userManager);
                break;
        }
        
        $this->updateScheduleCache($schedule);
        
        return $result;
    }
    
    private function detectScheduleChanges($schedule) {
        $scheduleId = $schedule['_id'];
        $cache = FileHandler::loadJSON(DATA_DIR . '/schedules/schedule_cache.json', []);
        
        if (!isset($cache[$scheduleId])) {
            return ['type' => 'new', 'old' => null];
        }
        
        $old = $cache[$scheduleId];
        
        if ($this->isCancelled($schedule)) {
            return ['type' => 'cancelled', 'old' => $old];
        }
        
        if (strtotime($schedule['startTime']) !== strtotime($old['startTime'])) {
            return ['type' => 'rescheduled', 'old' => $old];
        }
        
        return ['type' => 'unchanged', 'old' => $old];
    }
    
    private function isCancelled($schedule) {
        if (isset($schedule['status'])) {
            $status = strtoupper($schedule['status']);
            if ($status === 'CANCELLED' || $status === 'CANCELED') {
                return true;
            }
        }
        return false;
    }
    
    private function createNotificationSet($schedule) {
        $scheduleId = $schedule['_id'];
        $notifTimes = $this->calculateNotificationTimes($schedule);
        
        if (empty($notifTimes)) {
            return;
        }
        
        $pending = FileHandler::loadJSON(DATA_DIR . '/pending/pending_notifications.json', []);
        
        foreach ($notifTimes as $type => $time) {
            $key = $scheduleId . '_' . $type;
            $pending[$key] = [
                'schedule_id' => $scheduleId,
                'batch_id' => $schedule['batchId'] ?? '',
                'notification_type' => $type,
                'notification_time' => $time,
                'status' => 'pending',
                'created_at' => date('Y-m-d H:i:s')
            ];
        }
        
        FileHandler::saveJSON(DATA_DIR . '/pending/pending_notifications.json', $pending);
        $this->logger->info("    Created " . count($notifTimes) . " notifications");
    }
    
    private function calculateNotificationTimes($schedule) {
        $startTime = strtotime($schedule['startTime']);
        $times = [];
        
        foreach ($this->settings['notification_timings'] as $type => $config) {
            if (!$config['enabled']) continue;
            
            $offset = $config['offset_minutes'] * 60;
            $notifTime = $startTime + $offset;
            
            if ($notifTime > time()) {
                $times[$type] = $notifTime;
            }
        }
        
        return $times;
    }
    
    private function processPendingNotifications($schedule, $activeUsers, $userManager) {
        $scheduleId = $schedule['_id'];
        $batchId = $schedule['batchId'] ?? '';
        $pending = FileHandler::loadJSON(DATA_DIR . '/pending/pending_notifications.json', []);
        $currentTime = time();
        $sent = 0;
        
        foreach ($pending as $key => &$notif) {
            if ($notif['schedule_id'] !== $scheduleId || $notif['status'] !== 'pending') {
                continue;
            }
            
            if ($currentTime >= $notif['notification_time'] && 
                $currentTime <= ($notif['notification_time'] + 60)) {
                
                $this->logger->info("    → Sending: " . $notif['notification_type']);
                
                $sent += $this->sendNotifications($schedule, $activeUsers, $userManager, $notif['notification_type']);
                
                $notif['status'] = 'sent';
                $notif['sent_at'] = date('Y-m-d H:i:s');
            }
        }
        
        FileHandler::saveJSON(DATA_DIR . '/pending/pending_notifications.json', $pending);
        
        return $sent;
    }
    
    private function sendNotifications($schedule, $activeUsers, $userManager, $notifType) {
        $scheduleId = $schedule['_id'];
        $batchId = $schedule['batchId'] ?? '';
        $notifLog = FileHandler::loadJSON(DATA_DIR . '/notifications/notification_log.json', []);
        $sent = 0;
        
        foreach ($activeUsers as $telegramId => $userData) {
            if (!$userManager->shouldReceiveNotification($userData, $batchId, $notifType)) {
                continue;
            }
            
            if ($this->isDuplicate($notifLog, $telegramId, $scheduleId, $notifType)) {
                continue;
            }
            
            $message = $this->generateMessage($schedule, $notifType);
            
            $logEntry = [
                'notification_id' => uniqid('notif_', true),
                'telegram_id' => $telegramId,
                'batch_id' => $batchId,
                'schedule_id' => $scheduleId,
                'notification_type' => $notifType,
                'message_content' => $message,
                'message_id' => null,
                'status' => 'pending_send',
                'sent_at' => date('Y-m-d H:i:s'),
                'schedule_time' => $schedule['startTime'],
                'edit_history' => []
            ];
            
            $notifLog[] = $logEntry;
            $sent++;
            
            usleep($this->settings['rate_limiting']['batch_delay_ms'] * 1000);
        }
        
        FileHandler::saveJSON(DATA_DIR . '/notifications/notification_log.json', $notifLog);
        
        return $sent;
    }
    
    private function isDuplicate($notifLog, $telegramId, $scheduleId, $notifType) {
        foreach ($notifLog as $entry) {
            if ($entry['telegram_id'] === $telegramId &&
                $entry['schedule_id'] === $scheduleId &&
                $entry['notification_type'] === $notifType &&
                in_array($entry['status'], ['sent', 'pending_send'])) {
                return true;
            }
        }
        return false;
    }
    
    private function generateMessage($schedule, $notifType) {
        $config = $this->settings['notification_timings'][$notifType] ?? ['label' => 'Notification'];
        $label = $config['label'];
        
        $topic = $schedule['topic'] ?? 'Class';
        $subject = $schedule['subjectId']['name'] ?? 'Subject';
        $startTime = date('h:i A', strtotime($schedule['startTime']));
        $endTime = isset($schedule['endTime']) ? date('h:i A', strtotime($schedule['endTime'])) : 'N/A';
        
        $message = "<b>{$label}</b>\n\n";
        $message .= "📚 <b>Subject:</b> {$subject}\n";
        $message .= "📖 <b>Topic:</b> {$topic}\n";
        $message .= "⏰ <b>Time:</b> {$startTime} - {$endTime}\n";
        $message .= "📊 <b>Type:</b> " . ($schedule['lectureType'] ?? 'LIVE') . "\n";
        
        if (!empty($schedule['homeworkIds'])) {
            $message .= "📝 <b>Notes:</b> " . count($schedule['homeworkIds']) . " available\n";
        }
        
        return $message;
    }
    
    private function handleReschedule($schedule, $activeUsers, $userManager, $oldData) {
        $scheduleId = $schedule['_id'];
        
        $pending = FileHandler::loadJSON(DATA_DIR . '/pending/pending_notifications.json', []);
        foreach ($pending as $key => &$notif) {
            if ($notif['schedule_id'] === $scheduleId && $notif['status'] === 'pending') {
                $notif['status'] = 'cancelled';
            }
        }
        FileHandler::saveJSON(DATA_DIR . '/pending/pending_notifications.json', $pending);
        
        $notifLog = FileHandler::loadJSON(DATA_DIR . '/notifications/notification_log.json', []);
        $edited = 0;
        
        $oldTime = date('h:i A', strtotime($oldData['startTime']));
        $newTime = date('h:i A', strtotime($schedule['startTime']));
        
        foreach ($notifLog as &$entry) {
            if ($entry['schedule_id'] === $scheduleId && 
                $entry['status'] === 'sent' &&
                !empty($entry['message_id'])) {
                
                $entry['edit_history'][] = [
                    'edited_at' => date('Y-m-d H:i:s'),
                    'reason' => 'rescheduled',
                    'old_time' => $oldTime,
                    'new_time' => $newTime
                ];
                
                $entry['status'] = 'pending_edit';
                $entry['edit_message'] = $this->generateRescheduleMessage($schedule, $oldTime, $newTime);
                $edited++;
            }
        }
        
        FileHandler::saveJSON(DATA_DIR . '/notifications/notification_log.json', $notifLog);
        
        $this->createNotificationSet($schedule);
        
        if ($this->settings['notification_types']['rescheduled']) {
            $this->sendNotifications($schedule, $activeUsers, $userManager, 'rescheduled');
        }
        
        return $edited;
    }
    
    private function generateRescheduleMessage($schedule, $oldTime, $newTime) {
        $topic = $schedule['topic'] ?? 'Class';
        $subject = $schedule['subjectId']['name'] ?? 'Subject';
        
        $message = "⚠️ <b>CLASS RESCHEDULED</b>\n\n";
        $message .= "📚 <b>Subject:</b> {$subject}\n";
        $message .= "📖 <b>Topic:</b> {$topic}\n\n";
        $message .= "❌ <b>Old Time:</b> {$oldTime}\n";
        $message .= "✅ <b>New Time:</b> {$newTime}\n";
        
        return $message;
    }
    
    private function handleCancellation($schedule) {
        $scheduleId = $schedule['_id'];
        
        $pending = FileHandler::loadJSON(DATA_DIR . '/pending/pending_notifications.json', []);
        foreach ($pending as $key => &$notif) {
            if ($notif['schedule_id'] === $scheduleId && $notif['status'] === 'pending') {
                $notif['status'] = 'cancelled';
            }
        }
        FileHandler::saveJSON(DATA_DIR . '/pending/pending_notifications.json', $pending);
        
        $notifLog = FileHandler::loadJSON(DATA_DIR . '/notifications/notification_log.json', []);
        $cancelled = 0;
        
        foreach ($notifLog as &$entry) {
            if ($entry['schedule_id'] === $scheduleId && 
                $entry['status'] === 'sent' &&
                !empty($entry['message_id'])) {
                
                if ($this->settings['message_handling']['delete_on_cancel']) {
                    $entry['status'] = 'pending_delete';
                } else {
                    $entry['status'] = 'pending_edit';
                    $entry['edit_message'] = $this->generateCancellationMessage($schedule);
                }
                $cancelled++;
            }
        }
        
        FileHandler::saveJSON(DATA_DIR . '/notifications/notification_log.json', $notifLog);
        
        return $cancelled;
    }
    
    private function generateCancellationMessage($schedule) {
        $topic = $schedule['topic'] ?? 'Class';
        $subject = $schedule['subjectId']['name'] ?? 'Subject';
        
        $message = "🚫 <b>CLASS CANCELLED</b>\n\n";
        $message .= "📚 <b>Subject:</b> {$subject}\n";
        $message .= "📖 <b>Topic:</b> {$topic}\n\n";
        $message .= "❌ This class has been cancelled.\n";
        $message .= "📢 You'll be notified about rescheduling.";
        
        return $message;
    }
    
    private function updateScheduleCache($schedule) {
        $scheduleId = $schedule['_id'];
        $cache = FileHandler::loadJSON(DATA_DIR . '/schedules/schedule_cache.json', []);
        
        $cache[$scheduleId] = [
            '_id' => $schedule['_id'],
            'batchId' => $schedule['batchId'] ?? '',
            'topic' => $schedule['topic'] ?? '',
            'startTime' => $schedule['startTime'],
            'endTime' => $schedule['endTime'] ?? '',
            'status' => $schedule['status'] ?? '',
            'cached_at' => date('Y-m-d H:i:s')
        ];
        
        FileHandler::saveJSON(DATA_DIR . '/schedules/schedule_cache.json', $cache);
    }
}

// ================================================================
// LOGGER CLASS
// ================================================================

class Logger {
    private $logFile;
    private $logHandle;
    
    public function __construct() {
        $this->logFile = LOG_DIR . '/cron_' . date('Y-m-d') . '.log';
        $this->logHandle = fopen($this->logFile, 'a');
    }
    
    public function info($message, $context = []) {
        $this->log('INFO', $message, $context);
    }
    
    public function warning($message, $context = []) {
        $this->log('WARNING', $message, $context);
    }
    
    public function error($message, $context = []) {
        $this->log('ERROR', $message, $context);
    }
    
    public function debug($message, $context = []) {
        $this->log('DEBUG', $message, $context);
    }
    
    private function log($level, $message, $context = []) {
        $timestamp = date('Y-m-d H:i:s');
        $contextStr = empty($context) ? '' : ' ' . json_encode($context);
        $logLine = "[{$timestamp}] [{$level}] {$message}{$contextStr}\n";
        
        if ($this->logHandle) {
            fwrite($this->logHandle, $logLine);
        }
        
        if (php_sapi_name() === 'cli') {
            echo $logLine;
        }
    }
    
    public function close() {
        if ($this->logHandle) {
            fclose($this->logHandle);
        }
    }
}

?>