<?php
// sync_users.php
// Usage: php sync_users.php
// Requires: PHP with cURL enabled

date_default_timezone_set('Asia/Kolkata'); // set to your preferred timezone

$remoteUrl = 'https://api.titaniumtech.site/educational/v1/users.json';
// LOGS directory
$logsDir = __DIR__ . '/users_file_update';

// MAIN users.json file (save in same directory as this script)
$destFile = __DIR__ . '/users.json';
$tempFile = __DIR__ . '/users.json.tmp';

// runs.json should be inside logs directory
$runsFile = $logsDir . '/runs.json';

// ensure logs directory exists
if (!is_dir($logsDir)) {
    mkdir($logsDir, 0755, true);
}

// helper to write runs log atomically with locking
function append_run_log($runsFilePath, $entry) {
    $maxAttempts = 3;
    $attempt = 0;
    $success = false;
    while ($attempt < $maxAttempts && !$success) {
        $attempt++;
        $fp = fopen($runsFilePath, 'c+');
        if ($fp === false) {
            // try to create file
            if ($attempt === $maxAttempts) break;
            usleep(100000);
            continue;
        }
        // exclusive lock
        if (flock($fp, LOCK_EX)) {
            // read existing content
            $size = filesize($runsFilePath);
            $existing = $size ? fread($fp, $size) : '';
            $runs = [];
            if ($existing !== '') {
                $decoded = json_decode($existing, true);
                if (is_array($decoded)) {
                    $runs = $decoded;
                }
            }
            // append new entry at front
            array_unshift($runs, $entry);

            // optionally limit history length, e.g. keep 100 entries
            $maxHistory = 500;
            if (count($runs) > $maxHistory) {
                $runs = array_slice($runs, 0, $maxHistory);
            }

            // rewind and truncate
            ftruncate($fp, 0);
            rewind($fp);
            fwrite($fp, json_encode($runs, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
            fflush($fp);
            flock($fp, LOCK_UN);
            fclose($fp);
            $success = true;
            break;
        } else {
            fclose($fp);
            usleep(100000);
        }
    }
    return $success;
}

// Function to fetch remote content using cURL with robust error handling
function fetch_remote($url, &$httpCode = null, &$curlErr = null, $timeout = 20) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_MAXREDIRS, 5);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
    curl_setopt($ch, CURLOPT_USERAGENT, 'users-sync-script/1.0 (+https://yourdomain.example)');
    // For strict SSL validation leave as-is. If you face SSL problems, you can set CURLOPT_CAINFO or disable verify (not recommended).
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);

    $data = curl_exec($ch);
    if ($data === false) {
        $curlErr = curl_error($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        return false;
    }
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    return $data;
}

// Start run
$runStart = microtime(true);
$runTimestamp = date('c'); // ISO 8601 with timezone

$runEntry = [
    'timestamp' => $runTimestamp,
    'remote_url' => $remoteUrl,
    'local_file' => realpath($destFile) ?: $destFile,
    'status' => 'started',
    'http_code' => null,
    'bytes_received' => 0,
    'records_saved' => null,
    'duration_seconds' => null,
    'error' => null
];

$body = fetch_remote($remoteUrl, $httpCode, $curlErr, 30);
$runEntry['http_code'] = $httpCode;

if ($body === false) {
    $runEntry['status'] = 'failed';
    $runEntry['error'] = 'curl_error: ' . ($curlErr ?: 'unknown');
    $runEntry['duration_seconds'] = round(microtime(true) - $runStart, 3);
    append_run_log($runsFile, $runEntry);
    fwrite(STDERR, "Failed to fetch remote: " . $runEntry['error'] . PHP_EOL);
    exit(1);
}

// Validate it's JSON (optionally allow non-JSON passthrough)
$jsonDecoded = json_decode($body, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    // Not valid JSON — still save raw content but record error
    $runEntry['status'] = 'warning';
    $runEntry['error'] = 'invalid_json: ' . json_last_error_msg();
    // We will still save the body to file, but mark error
}

// Save to temporary file then rename (atomic)
$bytesWritten = file_put_contents($tempFile, $body, LOCK_EX);
if ($bytesWritten === false) {
    $runEntry['status'] = 'failed';
    $runEntry['error'] = 'failed_writing_temp_file';
    $runEntry['duration_seconds'] = round(microtime(true) - $runStart, 3);
    append_run_log($runsFile, $runEntry);
    fwrite(STDERR, "Failed to write temp file: $tempFile\n");
    exit(1);
}

// set permissions for the temp file (optional)
@chmod($tempFile, 0644);

// atomic rename
if (!rename($tempFile, $destFile)) {
    // fallback: try copying and unlinking
    if (!copy($tempFile, $destFile) || !unlink($tempFile)) {
        $runEntry['status'] = 'failed';
        $runEntry['error'] = 'failed_atomic_rename';
        $runEntry['duration_seconds'] = round(microtime(true) - $runStart, 3);
        append_run_log($runsFile, $runEntry);
        fwrite(STDERR, "Failed to move file to destination: $destFile\n");
        exit(1);
    }
}

// compute records_saved if JSON is array/object
$recordsSaved = null;
if (is_array($jsonDecoded)) {
    // If users.json is an array of users -> count; if object with key 'data' maybe count that
    if (array_keys($jsonDecoded) === range(0, count($jsonDecoded)-1)) {
        // sequential numeric keys => array
        $recordsSaved = count($jsonDecoded);
    } else {
        // associative object - try some heuristics
        if (isset($jsonDecoded['data']) && is_array($jsonDecoded['data'])) {
            $recordsSaved = count($jsonDecoded['data']);
        } else {
            $recordsSaved = 1; // object
        }
    }
}

// finalize run entry
$runEntry['status'] = ($runEntry['status'] === 'warning') ? 'completed_with_warnings' : 'completed';
$runEntry['bytes_received'] = (int)$bytesWritten;
$runEntry['records_saved'] = $recordsSaved;
$runEntry['duration_seconds'] = round(microtime(true) - $runStart, 3);

// append to runs.json
if (!append_run_log($runsFile, $runEntry)) {
    // If logging fails, write a small fallback file
    $fallback = $destDir . '/runs_log_fallback_' . time() . '.json';
    @file_put_contents($fallback, json_encode([$runEntry], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
}

// success message to stdout
echo "Sync completed: status={$runEntry['status']}, bytes={$runEntry['bytes_received']}, records={$runEntry['records_saved']}\n";
exit(0);
