<?php
/**
 * telegram_cron_detailed_report.php
 *
 * - Append every cron run to today's JSON log (runs_YYYY-MM-DD.json)
 * - Maintain one Telegram message per day, edited each run to include a numbered list:
 *     1) Run #1 — timestamp — status — duration — error (if any)
 *     2) Run #2 — ...
 * - Attempt to pin the message (if bot has permission).
 * - Safe for cron every 30 minutes. Use file locking and atomic writes.
 *
 * Usage: php telegram_cron_detailed_report.php
 */

date_default_timezone_set('Asia/Kolkata');

// ---------- CONFIG ----------
$botToken = '8508698990:AAHSUfRxWVEuuhlS5AvaJn0CUvuEvLa5kHY';
$chatId   = '6838875926';               // target chat id (user/group/channel)
$baseDir  = __DIR__ . '/users_file_update/telegram_cron';
@mkdir($baseDir, 0755, true);
// ----------------------------

$today = date('Y-m-d');
$dailyLogPath   = $baseDir . "/runs_$today.json";
$dailyMetaPath  = $baseDir . "/daily_report_meta_$today.json"; // stores { message_id, date, pinned:boolean, created_at }
$maxMessageChars = 3800; // leave margin under Telegram 4096 char limit

// Helper: request Telegram API method
function telegram_api($botToken, $method, $params = []) {
    $url = "https://api.telegram.org/bot{$botToken}/{$method}";
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    // send as form-data (sufficient for text)
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    curl_setopt($ch, CURLOPT_USERAGENT, 'telegram-cron-reporter/1.0');
    $res = curl_exec($ch);
    $err = curl_error($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    return [$res, $err, $http];
}

// Helper: atomic JSON read/write with lock
function read_json_file($path) {
    if (!file_exists($path)) return [];
    $fp = fopen($path, 'r');
    if (!$fp) return [];
    if (flock($fp, LOCK_SH)) {
        $content = stream_get_contents($fp);
        flock($fp, LOCK_UN);
    } else {
        $content = '';
    }
    fclose($fp);
    $decoded = json_decode($content, true);
    return is_array($decoded) ? $decoded : [];
}

function write_json_file_atomic($path, $data) {
    $tmp = $path . '.tmp';
    $encoded = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    if (file_put_contents($tmp, $encoded, LOCK_EX) === false) {
        return false;
    }
    // move into place
    if (!rename($tmp, $path)) {
        // try copy fallback
        if (!copy($tmp, $path)) return false;
        unlink($tmp);
    }
    return true;
}

// -------------------
// 1) Collect run info
// -------------------
$runStart = microtime(true);
$runEntry = [
    'run_number' => null, // will set after reading previous logs
    'timestamp' => date('c'),
    'time' => date('H:i:s'),
    'status' => 'success', // or 'error'
    'duration_ms' => null,
    'error' => null,
];

// Wrap cron work in try/catch. Put the code for your actual cron task where indicated.
try {
    // === PLACE YOUR CRON TASK HERE ===
    // Example: sync remote users file or any other job.
    // If something fails, throw new Exception('message')
    //
    // For demonstration we just simulate success. Remove simulation in production.
    //
    // throw new Exception('Simulated error'); // <-- uncomment to test error path
    //
    // =================================

    // No exception => success
} catch (Throwable $e) {
    $runEntry['status'] = 'error';
    $runEntry['error'] = $e->getMessage();
}

$runEntry['duration_ms'] = round((microtime(true) - $runStart) * 1000, 2);

// -------------------
// 2) Append to daily log
// -------------------
$logs = read_json_file($dailyLogPath);
$nextIndex = count($logs) + 1;
$runEntry['run_number'] = $nextIndex;
$logs[] = $runEntry;
write_json_file_atomic($dailyLogPath, $logs);

// -------------------
// 3) Ensure daily message exists (create if not) and pin if possible
// -------------------
$meta = read_json_file($dailyMetaPath);
// If meta exists but date mismatch, reset
if (!isset($meta['date']) || $meta['date'] !== $today) {
    $meta = [];
}

// Create message if missing
if (empty($meta['message_id'])) {
    // initial text (short)
    $initialText = "📌 *Daily Cron Report* — " . date('d M Y') . "\n\nInitializing report...";
    list($res, $err, $http) = telegram_api($botToken, 'sendMessage', [
        'chat_id' => $chatId,
        'text' => $initialText,
        'parse_mode' => 'Markdown'
    ]);
    $decoded = $res ? json_decode($res, true) : null;
    if ($decoded && isset($decoded['ok']) && $decoded['ok'] === true && isset($decoded['result']['message_id'])) {
        $meta['message_id'] = $decoded['result']['message_id'];
        $meta['date'] = $today;
        $meta['created_at'] = date('c');
        $meta['pinned'] = false;
        $meta['pin_error'] = null;
        write_json_file_atomic($dailyMetaPath, $meta);

        // Try to pin the message (best-effort). Works only if bot has permission in group/channel.
        list($pinRes, $pinErr, $pinHttp) = telegram_api($botToken, 'pinChatMessage', [
            'chat_id' => $chatId,
            'message_id' => $meta['message_id'],
            'disable_notification' => true
        ]);
        $pinDecoded = $pinRes ? json_decode($pinRes, true) : null;
        if ($pinDecoded && isset($pinDecoded['ok']) && $pinDecoded['ok'] === true) {
            $meta['pinned'] = true;
            $meta['pin_error'] = null;
        } else {
            // Save pin error (useful to show in report)
            $meta['pinned'] = false;
            $meta['pin_error'] = $pinErr ?: ($pinDecoded['description'] ?? "Pin failed, HTTP {$pinHttp}");
        }
        write_json_file_atomic($dailyMetaPath, $meta);
    } else {
        // failed to create message — save error details and continue
        $meta['message_id'] = null;
        $meta['date'] = $today;
        $meta['created_at'] = date('c');
        $meta['pinned'] = false;
        $meta['pin_error'] = 'sendMessage failed: ' . ($err ?: ($decoded['description'] ?? "HTTP {$http}"));
        write_json_file_atomic($dailyMetaPath, $meta);
        // We still continue — script will not throw
    }
}

// -------------------
// 4) Build message text with full numbered run list
// -------------------
function build_report_text($logs, $meta, $maxChars) {
    $header = "📌 *Daily Cron Report* — " . date('d M Y') . "\n";
    $header .= "_Last updated: " . date('H:i:s') . "_\n\n";
    $summary = "*Runs today*: " . count($logs) . "   |   *Pinned*: " . (isset($meta['pinned']) && $meta['pinned'] ? "Yes" : "No") . "\n\n";
    $body = "";
    foreach ($logs as $entry) {
        $num = $entry['run_number'] ?? '?';
        $time = $entry['time'] ?? '';
        $status = $entry['status'] === 'success' ? '🟢' : '🔴';
        $dur = (isset($entry['duration_ms']) ? $entry['duration_ms'] . 'ms' : '');
        $err = isset($entry['error']) && $entry['error'] ? " — `". str_replace('`', "'", $entry['error']) . "`" : '';
        $line = "{$num}. {$time} — {$status} — {$dur}{$err}\n";
        $body .= $line;
        // If body + header too long, break and note truncation
        if (mb_strlen($header . $summary . $body, 'UTF-8') > $maxChars) {
            $body .= "\n*… (output truncated in message; full log retained on server)*\n";
            break;
        }
    }
    return $header . $summary . $body;
}

$reportText = build_report_text($logs, $meta, $maxMessageChars);

// -------------------
// 5) Edit the existing message with the updated report
// -------------------
$editSucceeded = false;
if (!empty($meta['message_id'])) {
    list($editRes, $editErr, $editHttp) = telegram_api($botToken, 'editMessageText', [
        'chat_id' => $chatId,
        'message_id' => $meta['message_id'],
        'text' => $reportText,
        'parse_mode' => 'Markdown'
    ]);
    $editDecoded = $editRes ? json_decode($editRes, true) : null;
    if ($editDecoded && isset($editDecoded['ok']) && $editDecoded['ok'] === true) {
        $editSucceeded = true;
    } else {
        // edit error may occur if message was deleted or edited externally; clear message_id so next run recreates it
        $meta['edit_error'] = $editErr ?: ($editDecoded['description'] ?? "editMessageText failed HTTP {$editHttp}");
        // If the message doesn't exist or was deleted, reset message_id so we will recreate next run
        if (stripos($meta['edit_error'], 'message to edit not found') !== false || stripos($meta['edit_error'], 'message_id is invalid') !== false) {
            $meta['message_id'] = null;
        }
        write_json_file_atomic($dailyMetaPath, $meta);
    }
} else {
    // message missing: will be created on next run (we tried earlier but failed)
}

// -------------------
// 6) Save a short telegram-run log for debugging
// -------------------
$tgLogEntry = [
    'timestamp' => date('c'),
    'message_id' => $meta['message_id'] ?? null,
    'edit_success' => $editSucceeded,
    'meta' => $meta,
];

$tgLogPath = $baseDir . '/telegram_debug_log.json';
$tgLogs = read_json_file($tgLogPath);
array_unshift($tgLogs, $tgLogEntry);
// keep last 200 entries
if (count($tgLogs) > 200) $tgLogs = array_slice($tgLogs, 0, 200);
write_json_file_atomic($tgLogPath, $tgLogs);

// -------------------
// 7) Output for cron log and finish
// -------------------
echo "[" . date('c') . "] Run #{$runEntry['run_number']} saved. Telegram message " . ($meta['message_id'] ?? 'N/A') . ". Edit success: " . ($editSucceeded ? 'yes' : 'no') . ".\n";
if (isset($meta['pin_error']) && $meta['pin_error']) {
    echo "Pin error: " . $meta['pin_error'] . "\n";
}
if (isset($meta['edit_error']) && $meta['edit_error']) {
    echo "Edit error: " . $meta['edit_error'] . "\n";
}
exit(0);
