<?php
/**
 * ================================================================
 * TELEGRAM BOT API HANDLER
 * ================================================================
 * 
 * This file should be called by your Telegram bot to:
 * 1. Process pending_send notifications
 * 2. Edit pending_edit notifications
 * 3. Delete pending_delete notifications
 * 4. Update user bot_started status
 * 
 * Endpoint: telegram_api.php?action=ACTION
 * 
 * Actions:
 * - send_pending: Send all pending notifications
 * - process_webhook: Handle incoming Telegram updates
 * - update_user_status: Update user bot started status
 * 
 * @version 1.0.0
 */

// ================================================================
// CONFIGURATION
// ================================================================

define('BASE_DIR', __DIR__);
define('DATA_DIR', BASE_DIR . '/data');

// Load settings
$settings = loadJSON(DATA_DIR . '/settings.json', []);
$botToken = $settings['telegram']['bot_token'] ?? '';
$telegramAPI = "https://api.telegram.org/bot{$botToken}/";

// ================================================================
// MAIN ROUTING
// ================================================================

$action = $_GET['action'] ?? 'send_pending';

switch ($action) {
    case 'send_pending':
        sendPendingNotifications();
        break;
        
    case 'process_webhook':
        processWebhook();
        break;
        
    case 'update_user_status':
        updateUserStatus();
        break;
        
    case 'test':
        testConnection();
        break;
        
    default:
        respondJSON(['error' => 'Invalid action']);
}

// ================================================================
// SEND PENDING NOTIFICATIONS
// ================================================================

function sendPendingNotifications() {
    global $telegramAPI, $settings;
    
    $notifLog = loadJSON(DATA_DIR . '/notifications/notification_log.json', []);
    $processed = ['sent' => 0, 'edited' => 0, 'deleted' => 0, 'failed' => 0];
    
    foreach ($notifLog as &$entry) {
        // Send pending messages
        if ($entry['status'] === 'pending_send') {
            $result = sendMessage(
                $entry['telegram_id'],
                $entry['message_content']
            );
            
            if ($result['success']) {
                $entry['message_id'] = $result['message_id'];
                $entry['status'] = 'sent';
                $processed['sent']++;
            } else {
                $entry['status'] = 'failed';
                $entry['error'] = $result['error'];
                $processed['failed']++;
            }
            
            usleep($settings['rate_limiting']['batch_delay_ms'] * 1000);
        }
        
        // Edit messages
        if ($entry['status'] === 'pending_edit' && !empty($entry['message_id'])) {
            $result = editMessage(
                $entry['telegram_id'],
                $entry['message_id'],
                $entry['edit_message'] ?? $entry['message_content']
            );
            
            if ($result['success']) {
                $entry['status'] = 'edited';
                $processed['edited']++;
            } else {
                $entry['status'] = 'edit_failed';
                $entry['error'] = $result['error'];
                $processed['failed']++;
            }
            
            usleep($settings['rate_limiting']['batch_delay_ms'] * 1000);
        }
        
        // Delete messages
        if ($entry['status'] === 'pending_delete' && !empty($entry['message_id'])) {
            $result = deleteMessage(
                $entry['telegram_id'],
                $entry['message_id']
            );
            
            if ($result['success']) {
                $entry['status'] = 'deleted';
                $processed['deleted']++;
            } else {
                $entry['status'] = 'delete_failed';
                $entry['error'] = $result['error'];
                $processed['failed']++;
            }
            
            usleep($settings['rate_limiting']['batch_delay_ms'] * 1000);
        }
    }
    
    saveJSON(DATA_DIR . '/notifications/notification_log.json', $notifLog);
    
    respondJSON([
        'success' => true,
        'processed' => $processed
    ]);
}

// ================================================================
// PROCESS WEBHOOK (Bot Updates)
// ================================================================

function processWebhook() {
    $update = json_decode(file_get_contents('php://input'), true);
    
    if (!$update) {
        respondJSON(['error' => 'No update data']);
        return;
    }
    
    // Handle /start command
    if (isset($update['message']['text'])) {
        $text = $update['message']['text'];
        $chatId = $update['message']['chat']['id'];
        $username = $update['message']['chat']['username'] ?? '';
        
        if (strpos($text, '/start') === 0) {
            handleStartCommand($chatId, $username);
        }
        
        if (strpos($text, '/settings') === 0) {
            handleSettingsCommand($chatId);
        }
        
        if (strpos($text, '/stop') === 0) {
            handleStopCommand($chatId);
        }
    }
    
    // Handle callback queries (inline buttons)
    if (isset($update['callback_query'])) {
        handleCallbackQuery($update['callback_query']);
    }
    
    respondJSON(['success' => true]);
}

// ================================================================
// BOT COMMAND HANDLERS
// ================================================================

function handleStartCommand($chatId, $username) {
    // Update user status
    $userFile = DATA_DIR . "/users/{$chatId}.json";
    
    if (file_exists($userFile)) {
        $userData = loadJSON($userFile, []);
        $userData['bot_started'] = true;
        $userData['last_interaction'] = date('Y-m-d H:i:s');
        saveJSON($userFile, $userData);
        
        $message = "🎉 <b>Welcome back!</b>\n\n";
        $message .= "✅ Notifications are now <b>enabled</b>!\n\n";
        $message .= "You will receive notifications for:\n";
        $message .= "📅 New class schedules\n";
        $message .= "⏰ 1 hour before class\n";
        $message .= "🔔 10 minutes before class\n";
        $message .= "▶️ When class starts\n";
        $message .= "⚠️ Reschedules & cancellations\n\n";
        $message .= "Use /settings to customize your preferences.";
    } else {
        $message = "⚠️ <b>User not found!</b>\n\n";
        $message .= "Please contact admin to get access.\n";
        $message .= "Your Telegram ID: <code>{$chatId}</code>";
    }
    
    sendMessage($chatId, $message);
}

function handleStopCommand($chatId) {
    $userFile = DATA_DIR . "/users/{$chatId}.json";
    
    if (file_exists($userFile)) {
        $userData = loadJSON($userFile, []);
        $userData['global_notifications_enabled'] = false;
        $userData['last_interaction'] = date('Y-m-d H:i:s');
        saveJSON($userFile, $userData);
        
        $message = "🔕 <b>Notifications Disabled</b>\n\n";
        $message .= "You will no longer receive class notifications.\n\n";
        $message .= "Use /start to enable notifications again.";
    } else {
        $message = "⚠️ User not found!";
    }
    
    sendMessage($chatId, $message);
}

function handleSettingsCommand($chatId) {
    $userFile = DATA_DIR . "/users/{$chatId}.json";
    
    if (!file_exists($userFile)) {
        sendMessage($chatId, "⚠️ User not found!");
        return;
    }
    
    $userData = loadJSON($userFile, []);
    
    $message = "⚙️ <b>Your Notification Settings</b>\n\n";
    $message .= "🔔 Global Notifications: " . ($userData['global_notifications_enabled'] ? '✅ ON' : '❌ OFF') . "\n\n";
    
    $message .= "<b>Notification Types:</b>\n";
    $prefs = $userData['notification_preferences'] ?? [];
    $message .= "📅 Scheduled: " . ($prefs['scheduled'] ?? true ? '✅' : '❌') . "\n";
    $message .= "⏰ 1 Hour Before: " . ($prefs['before_1_hour'] ?? true ? '✅' : '❌') . "\n";
    $message .= "🔔 10 Min Before: " . ($prefs['before_10_min'] ?? true ? '✅' : '❌') . "\n";
    $message .= "▶️ Class Started: " . ($prefs['on_start'] ?? true ? '✅' : '❌') . "\n";
    $message .= "⚠️ Rescheduled: " . ($prefs['rescheduled'] ?? true ? '✅' : '❌') . "\n";
    $message .= "🚫 Cancelled: " . ($prefs['cancelled'] ?? true ? '✅' : '❌') . "\n\n";
    
    $message .= "<b>Your Batches:</b>\n";
    if (!empty($userData['batch_settings'])) {
        foreach ($userData['batch_settings'] as $batchId => $batchData) {
            $status = $batchData['notifications_enabled'] ? '✅' : '❌';
            $message .= "{$status} {$batchData['batch_name']}\n";
        }
    }
    
    $message .= "\n💡 Use the admin panel to customize settings.";
    
    // Send with inline keyboard for quick actions
    $keyboard = [
        [
            ['text' => '🔔 Toggle Notifications', 'callback_data' => 'toggle_global'],
            ['text' => '⚙️ Manage Batches', 'callback_data' => 'manage_batches']
        ]
    ];
    
    sendMessageWithKeyboard($chatId, $message, $keyboard);
}

function handleCallbackQuery($callbackQuery) {
    $chatId = $callbackQuery['message']['chat']['id'];
    $data = $callbackQuery['data'];
    
    if ($data === 'toggle_global') {
        $userFile = DATA_DIR . "/users/{$chatId}.json";
        if (file_exists($userFile)) {
            $userData = loadJSON($userFile, []);
            $userData['global_notifications_enabled'] = !($userData['global_notifications_enabled'] ?? true);
            saveJSON($userFile, $userData);
            
            $status = $userData['global_notifications_enabled'] ? 'enabled' : 'disabled';
            answerCallbackQuery($callbackQuery['id'], "Notifications {$status}!");
            handleSettingsCommand($chatId); // Refresh settings
        }
    }
}

// ================================================================
// TELEGRAM API FUNCTIONS
// ================================================================

function sendMessage($chatId, $text, $parseMode = 'HTML') {
    global $telegramAPI;
    
    $data = [
        'chat_id' => $chatId,
        'text' => $text,
        'parse_mode' => $parseMode
    ];
    
    $response = apiRequest('sendMessage', $data);
    
    if ($response && isset($response['ok']) && $response['ok']) {
        return [
            'success' => true,
            'message_id' => $response['result']['message_id'] ?? null
        ];
    }
    
    return [
        'success' => false,
        'error' => $response['description'] ?? 'Unknown error'
    ];
}

function sendMessageWithKeyboard($chatId, $text, $keyboard) {
    global $telegramAPI;
    
    $data = [
        'chat_id' => $chatId,
        'text' => $text,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode([
            'inline_keyboard' => $keyboard
        ])
    ];
    
    return apiRequest('sendMessage', $data);
}

function editMessage($chatId, $messageId, $text, $parseMode = 'HTML') {
    global $telegramAPI;
    
    $data = [
        'chat_id' => $chatId,
        'message_id' => $messageId,
        'text' => $text,
        'parse_mode' => $parseMode
    ];
    
    $response = apiRequest('editMessageText', $data);
    
    if ($response && isset($response['ok']) && $response['ok']) {
        return ['success' => true];
    }
    
    return [
        'success' => false,
        'error' => $response['description'] ?? 'Unknown error'
    ];
}

function deleteMessage($chatId, $messageId) {
    global $telegramAPI;
    
    $data = [
        'chat_id' => $chatId,
        'message_id' => $messageId
    ];
    
    $response = apiRequest('deleteMessage', $data);
    
    if ($response && isset($response['ok']) && $response['ok']) {
        return ['success' => true];
    }
    
    return [
        'success' => false,
        'error' => $response['description'] ?? 'Unknown error'
    ];
}

function answerCallbackQuery($callbackQueryId, $text) {
    $data = [
        'callback_query_id' => $callbackQueryId,
        'text' => $text
    ];
    
    return apiRequest('answerCallbackQuery', $data);
}

function apiRequest($method, $data) {
    global $telegramAPI;
    
    $ch = curl_init($telegramAPI . $method);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    $response = curl_exec($ch);
    curl_close($ch);
    
    return json_decode($response, true);
}

// ================================================================
// HELPER FUNCTIONS
// ================================================================

function loadJSON($filepath, $default = null) {
    if (!file_exists($filepath)) return $default;
    $content = file_get_contents($filepath);
    if ($content === false) return $default;
    $data = json_decode($content, true);
    return $data ?? $default;
}

function saveJSON($filepath, $data) {
    $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    $tempFile = $filepath . '.tmp';
    if (file_put_contents($tempFile, $json, LOCK_EX) === false) return false;
    return rename($tempFile, $filepath);
}

function respondJSON($data) {
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

function updateUserStatus() {
    $telegramId = $_POST['telegram_id'] ?? '';
    $started = $_POST['started'] ?? true;
    
    if (empty($telegramId)) {
        respondJSON(['error' => 'telegram_id required']);
        return;
    }
    
    $userFile = DATA_DIR . "/users/{$telegramId}.json";
    
    if (!file_exists($userFile)) {
        respondJSON(['error' => 'User not found']);
        return;
    }
    
    $userData = loadJSON($userFile, []);
    $userData['bot_started'] = (bool)$started;
    $userData['last_interaction'] = date('Y-m-d H:i:s');
    saveJSON($userFile, $userData);
    
    respondJSON(['success' => true]);
}

function testConnection() {
    global $telegramAPI;
    
    $response = apiRequest('getMe', []);
    
    if ($response && isset($response['ok']) && $response['ok']) {
        respondJSON([
            'success' => true,
            'bot_info' => $response['result']
        ]);
    } else {
        respondJSON([
            'success' => false,
            'error' => 'Failed to connect to Telegram API'
        ]);
    }
}

?>