<?php
/**
 * ================================================================
 * TEST NOTIFICATION GENERATOR
 * ================================================================
 * 
 * Usage:
 * - All notifications: test_notification.php
 * - Specific user by ID: test_notification.php?id=6838875926
 * - Specific user by username: test_notification.php?id=titanium
 * - Specific type: test_notification.php?type=scheduled
 * - Custom message: test_notification.php?custom=1
 * 
 * @version 1.0.0
 */

define('BASE_DIR', __DIR__);
define('DATA_DIR', BASE_DIR . '/data');

// ================================================================
// GET PARAMETERS
// ================================================================

$userId = $_GET['id'] ?? '6838875926'; // Default user
$notificationType = $_GET['type'] ?? 'all'; // all, scheduled, before_1_hour, before_10_min, on_start, rescheduled, cancelled
$customMessage = isset($_GET['custom']);

// ================================================================
// RESOLVE USER ID
// ================================================================

function resolveUserId($input) {
    // If it's numeric, it's a telegram_id
    if (is_numeric($input)) {
        return $input;
    }
    
    // Search by username
    $userFiles = glob(DATA_DIR . '/users/*.json');
    foreach ($userFiles as $file) {
        $userData = json_decode(file_get_contents($file), true);
        if (isset($userData['username']) && strtolower($userData['username']) === strtolower($input)) {
            return $userData['telegram_id'];
        }
        if (isset($userData['name']) && strtolower($userData['name']) === strtolower($input)) {
            return $userData['telegram_id'];
        }
    }
    
    return null;
}

$telegramId = resolveUserId($userId);

if (!$telegramId) {
    die("❌ User not found: $userId\n\nTry:\n- Numeric ID: ?id=6838875926\n- Username: ?id=titanium\n");
}

// Check if user exists
$userFile = DATA_DIR . "/users/{$telegramId}.json";
if (!file_exists($userFile)) {
    die("❌ User file not found for ID: $telegramId\n");
}

$userData = json_decode(file_get_contents($userFile), true);
$username = $userData['username'] ?? $userData['name'] ?? 'User';

echo "🎯 <b>Testing notifications for:</b> {$username} (ID: {$telegramId})\n\n";

// Check bot status
if (empty($userData['bot_started'])) {
    echo "⚠️  WARNING: User hasn't started the bot yet!\n";
    echo "   User must send /start to the bot first.\n\n";
}

if (!empty($userData['bot_blocked'])) {
    echo "⚠️  WARNING: User has blocked the bot!\n\n";
}

if (empty($userData['global_notifications_enabled'])) {
    echo "⚠️  WARNING: User has disabled notifications!\n\n";
}

// ================================================================
// LOAD EXISTING NOTIFICATION LOG
// ================================================================

$notifLogFile = DATA_DIR . '/notifications/notification_log.json';
$notifLog = file_exists($notifLogFile) ? json_decode(file_get_contents($notifLogFile), true) : [];

// ================================================================
// NOTIFICATION TEMPLATES
// ================================================================

$templates = [
    'scheduled' => [
        'label' => '📅 Class Scheduled',
        'message' => "<b>📅 Class Scheduled</b>\n\n" .
                    "📚 <b>Subject:</b> Physics\n" .
                    "📖 <b>Topic:</b> Centre of Mass 14: COM Frame || Variable Mass System\n" .
                    "⏰ <b>Time:</b> 03:30 PM - 05:30 PM\n" .
                    "📊 <b>Type:</b> LIVE\n" .
                    "👨‍🏫 <b>Teacher:</b> Amitabh Sir\n" .
                    "📝 <b>Notes:</b> Available after class\n\n" .
                    "🔔 You'll receive reminders before the class starts."
    ],
    'before_1_hour' => [
        'label' => '⏰ 1 Hour Reminder',
        'message' => "<b>⏰ Starting in 1 Hour!</b>\n\n" .
                    "📚 <b>Subject:</b> Physical Chemistry\n" .
                    "📖 <b>Topic:</b> Ionic Equilibrium 01: Factors Affecting Equilibria\n" .
                    "⏰ <b>Time:</b> 06:15 PM - 08:15 PM\n" .
                    "📊 <b>Type:</b> LIVE\n" .
                    "👨‍🏫 <b>Teacher:</b> Pankaj Sir\n\n" .
                    "⏳ Class starts in <b>60 minutes</b>. Get ready!"
    ],
    'before_10_min' => [
        'label' => '🔔 10 Minutes Reminder',
        'message' => "<b>🔔 Starting in 10 Minutes!</b>\n\n" .
                    "📚 <b>Subject:</b> Inorganic Chemistry\n" .
                    "📖 <b>Topic:</b> Chemical Bonding: VSEPR Theory\n" .
                    "⏰ <b>Time:</b> 07:00 PM - 08:30 PM\n" .
                    "📊 <b>Type:</b> LIVE\n" .
                    "👨‍🏫 <b>Teacher:</b> Pradeep Sir\n\n" .
                    "⚡ Class starts very soon! Join now!"
    ],
    'on_start' => [
        'label' => '▶️ Class Started',
        'message' => "<b>▶️ Class Started Now!</b>\n\n" .
                    "📚 <b>Subject:</b> Mathematics\n" .
                    "📖 <b>Topic:</b> Calculus: Limits and Derivatives\n" .
                    "⏰ <b>Time:</b> 08:00 PM - 10:00 PM\n" .
                    "📊 <b>Type:</b> LIVE\n" .
                    "👨‍🏫 <b>Teacher:</b> Vineet Sir\n\n" .
                    "🎥 <b>Class is LIVE now!</b> Join immediately!"
    ],
    'rescheduled' => [
        'label' => '⚠️ Class Rescheduled',
        'message' => "<b>⚠️ CLASS RESCHEDULED</b>\n\n" .
                    "📚 <b>Subject:</b> Organic Chemistry\n" .
                    "📖 <b>Topic:</b> Alkanes and Alkenes\n\n" .
                    "❌ <b>Old Time:</b> 05:00 PM\n" .
                    "✅ <b>New Time:</b> 06:30 PM\n\n" .
                    "📅 <b>Date:</b> " . date('d M Y') . "\n" .
                    "📊 <b>Type:</b> LIVE\n\n" .
                    "⏰ Please note the new timing!"
    ],
    'cancelled' => [
        'label' => '🚫 Class Cancelled',
        'message' => "<b>🚫 CLASS CANCELLED</b>\n\n" .
                    "📚 <b>Subject:</b> Biology\n" .
                    "📖 <b>Topic:</b> Cell Biology: Mitosis and Meiosis\n" .
                    "⏰ <b>Scheduled Time:</b> 04:00 PM - 06:00 PM\n" .
                    "📊 <b>Type:</b> LIVE\n\n" .
                    "❌ This class has been cancelled.\n" .
                    "📢 You'll be notified about rescheduling."
    ]
];

// ================================================================
// GENERATE TEST NOTIFICATIONS
// ================================================================

$typesToCreate = $notificationType === 'all' 
    ? array_keys($templates) 
    : [$notificationType];

$created = 0;

foreach ($typesToCreate as $type) {
    if (!isset($templates[$type])) {
        echo "❌ Invalid notification type: $type\n";
        continue;
    }
    
    $template = $templates[$type];
    
    // Get batch ID from user
    $batchId = '676e4dee1ec923bc192f38c9'; // Default
    if (!empty($userData['batch_settings'])) {
        $batches = array_keys($userData['batch_settings']);
        $batchId = $batches[0] ?? $batchId;
    }
    
    $testNotification = [
        'notification_id' => uniqid('test_' . $type . '_', true),
        'telegram_id' => $telegramId,
        'batch_id' => $batchId,
        'schedule_id' => 'test_schedule_' . time() . '_' . $type,
        'notification_type' => $type,
        'message_content' => $customMessage ? getCustomMessage($type) : $template['message'],
        'message_id' => null,
        'status' => 'pending_send',
        'sent_at' => date('Y-m-d H:i:s'),
        'schedule_time' => date('Y-m-d\TH:i:s.000\Z'),
        'edit_history' => [],
        'is_test' => true
    ];
    
    $notifLog[] = $testNotification;
    $created++;
    
    echo "✅ Created: {$template['label']} (ID: {$testNotification['notification_id']})\n";
}

// ================================================================
// SAVE NOTIFICATION LOG
// ================================================================

$saved = file_put_contents(
    $notifLogFile,
    json_encode($notifLog, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
);

if ($saved === false) {
    die("\n❌ Failed to save notification log!\n");
}

echo "\n📝 Total notifications created: {$created}\n";
echo "💾 Saved to: {$notifLogFile}\n\n";

// ================================================================
// AUTO-SEND OPTION
// ================================================================

$autoSend = isset($_GET['send']);

if ($autoSend) {
    echo "🚀 Auto-sending notifications...\n\n";
    
    $apiUrl = 'https://api.titaniumtech.site/educational/v1/notification/live/telegram_api.php?action=send_pending';
    
    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200 && $response) {
        $result = json_decode($response, true);
        
        if (isset($result['success']) && $result['success']) {
            echo "✅ Notifications sent successfully!\n";
            if (isset($result['processed'])) {
                echo "   📤 Sent: {$result['processed']['sent']}\n";
                echo "   ✏️  Edited: {$result['processed']['edited']}\n";
                echo "   🗑️  Deleted: {$result['processed']['deleted']}\n";
                echo "   ❌ Failed: {$result['processed']['failed']}\n";
            }
        } else {
            echo "⚠️  API returned error: " . ($result['error'] ?? 'Unknown') . "\n";
        }
    } else {
        echo "❌ Failed to send notifications (HTTP {$httpCode})\n";
    }
} else {
    echo "📤 <b>To send these notifications, run:</b>\n";
    echo "   curl https://api.titaniumtech.site/educational/v1/notification/live/telegram_api.php?action=send_pending\n\n";
    echo "   Or visit this URL with &send=1 to auto-send:\n";
    echo "   " . $_SERVER['REQUEST_URI'] . (strpos($_SERVER['REQUEST_URI'], '?') ? '&' : '?') . "send=1\n";
}

// ================================================================
// STATISTICS
// ================================================================

echo "\n" . str_repeat("=", 60) . "\n";
echo "📊 <b>User Statistics</b>\n";
echo str_repeat("=", 60) . "\n";
echo "👤 Username: {$username}\n";
echo "🆔 Telegram ID: {$telegramId}\n";
echo "🤖 Bot Started: " . ($userData['bot_started'] ? '✅ Yes' : '❌ No') . "\n";
echo "🔔 Notifications: " . ($userData['global_notifications_enabled'] ? '✅ Enabled' : '❌ Disabled') . "\n";
echo "📚 Active Batches: " . count($userData['batch_settings'] ?? []) . "\n";
echo "📨 Total Received: " . ($userData['total_notifications_received'] ?? 0) . "\n";
echo str_repeat("=", 60) . "\n";

// ================================================================
// USAGE EXAMPLES
// ================================================================

echo "\n💡 <b>Usage Examples:</b>\n";
echo str_repeat("-", 60) . "\n";
echo "All notifications for default user:\n";
echo "  test_notification.php\n\n";
echo "Specific user by ID:\n";
echo "  test_notification.php?id=6838875926\n\n";
echo "Specific user by username:\n";
echo "  test_notification.php?id=titanium\n\n";
echo "Specific notification type:\n";
echo "  test_notification.php?type=scheduled\n";
echo "  test_notification.php?type=before_1_hour\n";
echo "  test_notification.php?type=rescheduled\n\n";
echo "Auto-send after creating:\n";
echo "  test_notification.php?id=titanium&send=1\n\n";
echo "Custom message:\n";
echo "  test_notification.php?custom=1\n";
echo str_repeat("-", 60) . "\n";

// ================================================================
// CUSTOM MESSAGE GENERATOR
// ================================================================

function getCustomMessage($type) {
    $subjects = ['Physics', 'Chemistry', 'Mathematics', 'Biology'];
    $topics = [
        'Physics' => ['Mechanics', 'Thermodynamics', 'Electromagnetism', 'Optics'],
        'Chemistry' => ['Organic Chemistry', 'Inorganic Chemistry', 'Physical Chemistry'],
        'Mathematics' => ['Calculus', 'Algebra', 'Trigonometry', 'Coordinate Geometry'],
        'Biology' => ['Cell Biology', 'Genetics', 'Ecology', 'Human Physiology']
    ];
    $teachers = ['Amitabh Sir', 'Pankaj Sir', 'Vineet Sir', 'Pradeep Sir'];
    
    $subject = $subjects[array_rand($subjects)];
    $topic = $topics[$subject][array_rand($topics[$subject])];
    $teacher = $teachers[array_rand($teachers)];
    
    $startTime = date('h:i A', time() + 3600);
    $endTime = date('h:i A', time() + 7200);
    
    switch ($type) {
        case 'scheduled':
            return "<b>📅 Class Scheduled</b>\n\n" .
                   "📚 <b>Subject:</b> {$subject}\n" .
                   "📖 <b>Topic:</b> {$topic}\n" .
                   "⏰ <b>Time:</b> {$startTime} - {$endTime}\n" .
                   "👨‍🏫 <b>Teacher:</b> {$teacher}\n" .
                   "📊 <b>Type:</b> LIVE\n\n" .
                   "🔔 Custom test notification!";
                   
        case 'before_1_hour':
            return "<b>⏰ Starting in 1 Hour!</b>\n\n" .
                   "📚 <b>Subject:</b> {$subject}\n" .
                   "📖 <b>Topic:</b> {$topic}\n" .
                   "⏰ <b>Time:</b> {$startTime} - {$endTime}\n" .
                   "👨‍🏫 <b>Teacher:</b> {$teacher}\n\n" .
                   "⏳ Custom reminder - Class starts soon!";
                   
        case 'before_10_min':
            return "<b>🔔 Starting in 10 Minutes!</b>\n\n" .
                   "📚 <b>Subject:</b> {$subject}\n" .
                   "📖 <b>Topic:</b> {$topic}\n" .
                   "⏰ <b>Time:</b> {$startTime} - {$endTime}\n\n" .
                   "⚡ Custom alert - Join now!";
                   
        case 'on_start':
            return "<b>▶️ Class Started Now!</b>\n\n" .
                   "📚 <b>Subject:</b> {$subject}\n" .
                   "📖 <b>Topic:</b> {$topic}\n" .
                   "👨‍🏫 <b>Teacher:</b> {$teacher}\n\n" .
                   "🎥 Custom notification - LIVE now!";
                   
        case 'rescheduled':
            return "<b>⚠️ CLASS RESCHEDULED</b>\n\n" .
                   "📚 <b>Subject:</b> {$subject}\n" .
                   "📖 <b>Topic:</b> {$topic}\n\n" .
                   "❌ <b>Old Time:</b> 05:00 PM\n" .
                   "✅ <b>New Time:</b> {$startTime}\n\n" .
                   "⏰ Custom reschedule alert!";
                   
        case 'cancelled':
            return "<b>🚫 CLASS CANCELLED</b>\n\n" .
                   "📚 <b>Subject:</b> {$subject}\n" .
                   "📖 <b>Topic:</b> {$topic}\n" .
                   "⏰ <b>Was scheduled:</b> {$startTime}\n\n" .
                   "❌ Custom cancellation notice";
    }
    
    return "Test notification";
}

?>